<?php

declare(strict_types=1);

namespace MageQL;

use Exception;
use Mage;
use Mage_Core_Model_Abstract;
use Mage_Eav_Model_Entity_Attribute_Source_Abstract;
use Varien_Object;
use Zend_Log;

use GraphQL\Type\Definition\ResolveInfo;

/**
 * Forwards the source to the next resolver.
 *
 * @template T
 * @param T $a
 * @return T
 */
function forwardResolver($a) {
    return $a;
}

/**
 * @param string $str
 */
function snakeCaseToCamel($str): string {
    // TODO: Cache?
    return lcfirst(str_replace("_", "", ucwords($str, "_")));
}

/**
 * @param string $str
 */
function spacedToCamel($str): string {
    return str_replace([" ", "-"], ["", "_"], $str);
}

/**
 * @param string $str
 */
function camelToSnakeCase($str): string {
    // TODO: Can we reuse the Varien_Object cache somehow?
    return strtolower(preg_replace("/(.)([A-Z])/", "$1_$2", $str));
}

/**
 * @param mixed $unusedContext
 * @return mixed
 */
function defaultVarienObjectResolver(Varien_Object $src, array $unusedArgs, $unusedContext, ResolveInfo $info) {
    $getter = "get".ucfirst($info->fieldName);
    $has = "has".ucfirst($info->fieldName);

    if(method_exists($src, $getter) || $src->$has()) {
        return $src->$getter();
    }

    return null;
}

/**
 * @param mixed $context
 */
function dateAttributeResolver(Varien_Object $src, array $args, $context, ResolveInfo $info): ?string {
    $value = defaultVarienObjectResolver($src, $args, $context, $info);

    if($value !== null) {
        $parsed = strtotime($value);

        return gmdate("Y-m-d\TH:i:s\Z", $parsed);
    }

    return null;
}

function attributeSourceModel(Mage_Core_Model_Abstract $src, string $name): ?Mage_Eav_Model_Entity_Attribute_Source_Abstract {
    $attributeCode = camelToSnakeCase($name);

    // Essentially copied from Mage_Catalog_Model_Product::getAttributeText
    // This is missing on Categories and Customer, and here modified to also support
    // multiselect with array-storage
    /** @var \Mage_Eav_Model_Entity_Abstract $resource */
    $resource = $src->getResource();

    $attrResource = $resource->getAttribute($attributeCode);

    // If the attribute no longer exists
    if( ! $attrResource) {
        $error = sprintf(
            "%s: Could not load attribute resource model for '%s' of entity %s in store %d.",
            __FUNCTION__,
            $attributeCode,
            get_class($src),
            Mage::app()->getStore()->getId()
        );

        if(Mage::getIsDeveloperMode()) {
            throw new Exception($error);
        }
        else {
            Mage::log($error, Zend_Log::ERR);
        }

        return null;
    }

    return $attrResource->getSource();
}

/**
 * @param mixed $context
 */
function selectAttributeResolver(Mage_Core_Model_Abstract $src, array $args, $context, ResolveInfo $info): ?string {
    $value = defaultVarienObjectResolver($src, $args, $context, $info);
    $model = attributeSourceModel($src, $info->fieldName);

    if( ! $model) {
        return null;
    }

    return $model->getOptionText($value) ?: $value;
}

/**
 * @param mixed $args
 * @param mixed $context
 */
function multiselectAttributeResolver(Mage_Core_Model_Abstract $src, $args, $context, ResolveInfo $info): ?array {
    $value = defaultVarienObjectResolver($src, $args, $context, $info);
    $model = attributeSourceModel($src, $info->fieldName);

    if( ! $model) {
        return null;
    }

    return array_filter(array_map([$model, "getOptionText"], array_map("trim", is_array($value) ? $value : explode(",", $value))));
}
