<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Core_Model_Customer extends Mage_Core_Model_Abstract {
    const NOT_MODIFIED = "notModified";

    /**
     * @param mixed $unusedSrc
     */
    public static function resolveCustomer(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $unusedCtx,
        ResolveInfo $info
    ): ?Mage_Customer_Model_Customer {
        $session = Mage::getSingleton("customer/session");

        if($session->isLoggedIn()) {
            return $session->getCustomer();
        }

        return null;
    }

    public static function resolveCreatedAt(
        Mage_Customer_Model_Customer $src
    ): string {
        return gmdate("Y-m-d\TH:i:s\Z", strtotime($src->getCreatedAt()));
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function mutateLogin(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): MageQL_Core_Model_Customer_Result_Login {
        $email = $args["email"];
        $password = $args["password"];

        try {
            // TODO: Should these be a part of the context?
            $session = Mage::getSingleton("customer/session");
            // TODO: Attributes?
            $customer = Mage::getModel("customer/customer")
                ->setStore($ctx->getStore())
                ->loadByEmail($email);

            $session->login($email, $password);
            $session->setCustomerAsLoggedIn($customer);
            $customer->setSession($session->getSessionId());

            return new MageQL_Core_Model_Customer_Result_Login(MageQL_Core_Model_Customer_Result_Login::SUCCESS, $customer);
        }
        catch(Mage_Core_Exception $e) {
            if($e->getMessage() === Mage::helper("customer")->__("Invalid login or password.")) {
                return new MageQL_Core_Model_Customer_Result_Login(MageQL_Core_Model_Customer_Result_Login::ERROR_INVALID_LOGIN);
            }

            if($e->getMessage() === Mage::helper("customer")->__("This account is not confirmed.")) {
                return new MageQL_Core_Model_Customer_Result_Login(MageQL_Core_Model_Customer_Result_Login::ERROR_NOT_CONFIRMED);
            }

            throw $e;
        }
    }

    public static function mutateLogout(): bool {
        $session = Mage::getSingleton("customer/session");

        if($session->isLoggedIn()) {
            $session->logout();

            return true;
        }

        return false;
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function mutateUpdateEmail(
        $unusedSrc,
        array $args
    ): MageQL_Core_Model_Customer_Result_UpdateEmail {
        $session = Mage::getSingleton("customer/session");
        $email = trim($args["email"]);

        if( ! $session->isLoggedIn()) {
            return new MageQL_Core_Model_Customer_Result_UpdateEmail(MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_NOT_LOGGED_IN);
        }

        $customer = $session->getCustomer();

        if( ! Zend_Validate::is($email, "EmailAddress")) {
            return new MageQL_Core_Model_Customer_Result_UpdateEmail(MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_INVALID_EMAIL_ADDRESS, $customer);
        }

        if($email === $customer->getEmail()) {
            return new MageQL_Core_Model_Customer_Result_UpdateEmail(MageQL_Core_Model_Customer_Result_UpdateEmail::NOT_MODIFIED, $customer);
        }

        // Save the old email so we can notify customer
        $customer->setOldEmail($customer->getEmail());
        $customer->setEmail($email);
        $customer->setRpToken(null);
        $customer->setRpTokenCreatedAt(null);

        try {
            $customer->save();
        }
        catch(Mage_Customer_Exception $e) {
            if($e->getCode() === Mage_Customer_Model_Customer::EXCEPTION_EMAIL_EXISTS) {
                return new MageQL_Core_Model_Customer_Result_UpdateEmail(MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_EMAIL_EXISTS, $customer);
            }

            throw $e;
        }

        $customer->sendChangedPasswordOrEmail();

        // Ensure we update the email in the quote if any
        $checkout = Mage::getSingleton("checkout/session");

        if($checkout->hasQuote()) {
            $quote = $checkout->getQuote();

            $quote->setCustomer($customer);
            $quote->save();
        }

        return new MageQL_Core_Model_Customer_Result_UpdateEmail(MageQL_Core_Model_Customer_Result_UpdateEmail::SUCCESS, $customer);
    }
}
