# Crossroads GraphQL Base Module

Provides routing, configuration and utilties for building GraphQL APIs.

## Registering types and fields

```xml
<config>
  <default>
    <mageql>
      <schema>
        <default> <!-- The name of the schema to add the types to -->
          <ModuleName>
            <model>modulename/schemamodel</model>
          </ModuleName>
        </default>
      </schema>
    </mageql>
  </default>
</config>
```

```php
<?php

declare(strict_types=1);

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

class Modulename_Model_Schemamodel extends MageQL_Model_Schema_Abstract {
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        // Here we define the types we support:
        case "MyType":
            return $this->object("The object type we have added");
        }

        // If $typeName is not a type defined in this schema model:
        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        // Here we define the fields of any object or interface types. We can
        // also define fields for types not defined in this modue which will
        // extend them:
        case "MyType":
            return [
                // Field of type string, uses default getter variant to obtain
                // the value:
                "myfield" => $this->field("String", "A string-field in MyType"),
                "another" => $this->field("Int", "Another field in MyType")
                    ->setResolver(...),
            ];

        case "Query":
            return [
                "mytype" => $this->field("Mytype"),
            ];
        }

        return [];
    }
}
```

### Implementing an interface

```php
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "MyInterface":
            return $this->interface("An interface");

        case "MyType":
            return $this->object("The object type we have added")
                ->setInterfaces(["MyInterface"]);
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "MyInterface":
            // The interface fields, can contain default resolvers and so on:
            return [
                "myfield" => $this->field("String", "A string-field in MyType"),
                "another" => $this->field("Int", "Another field in MyType")
                    ->setResolver("resolveAnother"),
            ];

        case "MyType":
            // We use the $registry to fetch the other fields from the
            // interface so we do not have to redefine them:
            return array_merge($registry->getFieldBuilders("MyInterface"), [
                // Fields only existing on this type:
                "mytypefield" => $this->field("String!", "This must exist on MyType"),
            ]);
        }

        return [];
    }
```
