<?php

declare(strict_types=1);

use Spatie\Snapshots\MatchesSnapshots;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class RequestTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testMissingGetParameters() {
        // Test without developer mode for fails to avoid a stack trace
        Mage::setIsDeveloperMode(false);

        $resp = MagentoManager::runRequest(new Request("GET /graphql", []));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals("", MagentoManager::getSystemLog());
        $this->assertEquals("", MagentoManager::getExceptionLog());
    }

    public function testEmptyPostRequest() {
        Mage::setIsDeveloperMode(false);
        $resp = MagentoManager::runRequest(new Request("POST /graphql", []));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals("", MagentoManager::getSystemLog());
        $this->assertEquals("", MagentoManager::getExceptionLog());
    }

    public function testEmptyQueryGraphQL() {
        Mage::setIsDeveloperMode(false);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], ""));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals("", MagentoManager::getSystemLog());
        $this->assertEquals("", MagentoManager::getExceptionLog());
    }

    public function testGrpahQLQuery() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], "query { info { baseUrl } }"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testJSONQuery() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], '{"query":"query { info { baseUrl } }"}'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testMultiQuerySingle() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], '[{"query":"query { info { baseUrl } }"}]'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testMultiQuery() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], '[{"query":"query { info { baseUrl } }"}, {"query": "query { info { name } }"}]'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testMultiQueryError() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], '[{"query":"query { info { baseUrl } }"}, {"query": "query { doesNotExist { name } }"}]'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testNoGraphQL() {
        $resp = MagentoManager::runRequest(new Request("GET /", []));

        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertNotEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testMissingType() {
        Mage::setIsDeveloperMode(false);
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], '{
        "query": "query ($foo: MissingTypeHere) { info { baseUrl } }",
        "variables": "{\"foo\":{}}"
    }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
