<?php

declare(strict_types=1);

namespace MageQL;

use GraphQL\Type\Definition\Type;
use GraphQL\Type\Schema;

use MageQL\Schema\AbstractSchema;
use MageQL\Type\AbstractBuilder;

/**
 * Dynamically expanding type registry.
 */
class Registry {
    protected $includeUnreachable = false;
    protected $schema = null;
    protected $factories = [];

    /**
     * If set to true it will cause the Registry to load all unreachable types
     * provided by Schemas. By default this is false.
     */
    public function setIncludeUnreachable(bool $includeUnreachable) {
        $this->includeUnreachable = $includeUnreachable;

        return $this;
    }

    /**
     * Registers a new type factory.
     */
    public function addFactory(AbstractSchema $factory): self {
        $this->factories[] = $factory;

        return $this;
    }

    /**
     * Retrieves the type specified by the given name.
     */
    public function getType(string $typeName): Type {
        return $this->schema->getType($typeName);
    }

    /**
     * Creates a type, includes array and strict types.
     */
    protected function createType(string $typeName): Type {
        if(substr($typeName, -1) === "!") {
            return $this->wrapNonNull($this->schema->getType(substr($typeName, 0, -1)));
        }
        else if(substr($typeName, 0, 1) === "[") {
            if(substr($typeName, -1) !== "]") {
                throw new TypeException(sprintf("Missing closing ']' in type '%s'.", $typeName));
            }

            return $this->wrapListOf($this->schema->getType(substr($typeName, 1, -1)));
        }
        else {
            return $this->createTypeInstance($typeName);
        }
    }

    /**
     * Returns a map of field-builders for the given type.
     *
     * @return Array<string, FieldBuilder>
     */
    public function getFieldBuilders(string $typeName): array {
        $fields = [];

        foreach($this->factories as $factory) {
            // TODO: Check if the values are actually FieldBuilders?
            $fields = array_merge($fields, $factory->getTypeFields($typeName, $this));

        }

        return $fields;
    }

    public function getFields(string $typeName): array {
        return array_map(function($field) {
            return $field->createInstance($this);
        }, $this->getFieldBuilders($typeName));
    }

    /**
     * Creates a scheam using the added factories, with the types Query and
     * Mutation as base-types.
     */
    public function createSchema(): Schema {
        if($this->schema) {
            throw new Exception(sprintf("%s: Cannot create schema twice", __METHOD__));
        }

        $unreachable = $this->includeUnreachable ? $this->createUnreachableTypes() : [];

        $this->schema = new Schema([
            "query" => $this->createType("Query"),
            "mutation" => $this->createType("Mutation"),
            "types" => $unreachable,
            "typeLoader" => function($type) {
                return $this->createType($type);
            },
        ]);

        return $this->schema;
    }

    protected function createUnreachableTypes() {
        $types = [];
        $typeNames = [];

        foreach($this->factories as $factory) {
            $typeNames = array_merge($typeNames, $factory->getUnreachableTypes($this));
        }

        foreach($typeNames as $name) {
            $types[] = $this->createTypeInstance($name);
        }

        return $types;
    }

    protected function createTypeInstance(string $typeName): Type {
        foreach($this->factories as $factory) {
            $type = $factory->getTypeBuilder($typeName, $this);

            if($type) {
                $type->setName($typeName);

                return $type->createInstance($this);
            }
        }

        throw new MissingTypeException($typeName);
    }

    protected function wrapNonNull(Type $type): Type {
        $wrapped = Type::nonNull($type);

        $wrapped->name = $type->name . "!";

        return $wrapped;
    }

    protected function wrapListOf(Type $type): Type {
        $wrapped = Type::listOf($type);

        $wrapped->name = "[" . $type->name . "]";

        return $wrapped;
    }
}
