<?php

use GraphQL\Error\ClientAware;
use GraphQL\Error\Debug;
use GraphQL\Error\Error;
use GraphQL\Error\FormattedError;
use GraphQL\Error\Warning;
use GraphQL\GraphQL;
use GraphQL\Server\Helper;
use GraphQL\Server\RequestError;
use GraphQL\Server\StandardServer;
use GraphQL\Type\Schema;
use GraphQL\Utils\Utils;

use MageQL\Registry;
use MageQL\NotInstalledException;
use MageQL\InactiveStoreException;
use MageQL\SessionFailedException;
use MageQL\Schema\DefaultSchema;

class MageQL_Core_Router_GraphQL extends Mage_Core_Controller_Varien_Router_Abstract {
    const GRAPHQL_AREA = "frontend";
    const GRAPHQL_PATH = "graphql";
    const GRAPHIQL_PATH = "graphiql";
    const SCHEMA_DEFAULT = "default";

    public function collectRoutes($_area, $_useNames) {
        // Intentionally left empty
    }

    /**
     * Session namespace to refer in other places
     */
    public function getSessionNamespace() {
        $namespace = trim(strtolower(Mage::getStoreConfig("web/cookie/frontend_namespace")));

        if(empty($namespace)) {
            $namespace = self::DEFAULT_SESSION_NAMESPACE;
        }

        if($namespace == Mage_Adminhtml_Controller_Action::SESSION_NAMESPACE) {
            throw new RuntimeException(sprintf("%s: Session namespace matches admin namespace '%s'.", __METHOD__, Mage_Adminhtml_Controller_Action::SESSION_NAMESPACE));
        }

        return $namespace;
    }

    public function match(Zend_Controller_Request_Http $request): bool {
        $path   = array_filter(explode("/", trim($request->getPathInfo(), "/")));
        $front  = $this->getFront();
        $type   = $path[0] ?? null;
        $schema = $path[1] ?? null;

        switch($type) {
        case self::GRAPHIQL_PATH:
            return $this->graphiQL($request, $front->getResponse(), $schema);
        case self::GRAPHQL_PATH:
            return $this->graphQL($request, $front->getResponse(), $schema);
        default:
            return false;
        }
    }

    public function graphiQL(Zend_Controller_Request_Http $request, Zend_Controller_Response_Abstract $response, $schemaName = null): bool {
        $block = Mage::getSingleton("core/layout")
            ->createBlock("mageql/graphiql")
            ->setSchemaName($schemaName);

        $response->setBody($block->toHtml());

        $request->setDispatched(true);

        return true;
    }

    /**
     * Writes the supplied `$result` to `$response` and marks `$request` as
     * dispatched, `$code` will be set as the response code if it is larger
     * than the already set code on `$response`.
     */
    public function setJsonResponse(Zend_Controller_Request_Http $request, Zend_Controller_Response_Abstract $response, $result, int $code = 200): bool {
        // Only overwrite the response code if we have not already set one,
        // or if we get an exception/user-error
        if($response->getHttpResponseCode() < $code) {
            $response->setHttpResponseCode($code);
        }

        $jsonFlags = JSON_PRESERVE_ZERO_FRACTION // Preserve the floats
            | JSON_UNESCAPED_SLASHES // Escaped slashes is an annoyance and a non-standard part of JSON
            | JSON_UNESCAPED_UNICODE // We send full unicode encoded anyway
            | (Mage::getIsDeveloperMode() ? JSON_PRETTY_PRINT : 0); // Make it easy to read

        $response->setHeader("Content-Type", "application/json; charset=utf-8", true)
                 ->setBody(json_encode($result, $jsonFlags));

        $request->setDispatched(true);

        return true;
    }

    public function graphQL(
        Zend_Controller_Request_Http $request,
        Zend_Controller_Response_Abstract $response,
        $schemaName = null
    ): bool {
        try {
            $app = Mage::app();
            $store = $app->getStore();
            $session = $this->initSession($request, $response, $app, $store);

            // TODO: Create a context which wraps cart, session and store in one

            // Use experimental executor for better performance
            GraphQL::useExperimentalExecutor();

            $context = Mage::getModel("mageql/context", [
                "store"   => $store,
                "session" => $session,
            ]);

            $debug = Mage::getIsDeveloperMode() ? Debug::INCLUDE_DEBUG_MESSAGE | Debug::INCLUDE_TRACE : false;
            // Always include unreachable if we are in debug or if we are querying from GraphiQL
            $includeUnreachable = Mage::getIsDeveloperMode() || strpos($request->getHeader("Referer"), self::GRAPHIQL_PATH) !== false;
            $schema = $this->loadSchema($context, $schemaName ?: self::SCHEMA_DEFAULT, $includeUnreachable);
            $server = new StandardServer([
                "schema" => $schema,
                "rootValue" => $store,
                "context" => $context,
                "queryBatching" => true,
                "errorsHandler" => [$this, "handleError"],
                "debug" => $debug,
            ]);

            $data = $this->parseHttpRequest($request, $server->getHelper());

            $result = $server->executeRequest($data);
        }
        // RequestError is ClientAware
        catch(ClientAware $e) {
            return $this->setJsonResponse($request, $response, [
                "errors" => [
                    FormattedError::createFromException($e, Mage::getIsDeveloperMode()),
                ],
            ], 400);
        }
        // TODO: Handle session failure
        // TODO: Handle these?
        /*catch(Exception $e) {
            Mage::logException($e);

            // TODO: Debug stacktrace in debug mode
            return $this->setJsonResponse($request, $response, [Error::createLocatedError($e)], 500);
        }*/

        $responseCode = $this->getResponseCode($result);
        $respData = is_array($result) ? array_map(function($r) use($debug) {
            return $r->toArray($debug);
        }, $result) : $result->toArray($debug);

        return $this->setJsonResponse($request, $response, $respData, $responseCode);
    }

    public function initSession(
        Zend_Controller_Request_Http $request,
        Zend_Controller_Response_Abstract $response,
        Mage_Core_Model_App $app,
        Mage_Core_Model_Store $store
    ) {
        // Adapted from Mage_Core_Controller_Varien_Action::preDispatch()
        if( ! Mage::isInstalled()) {
            throw new NotInstalledException();
        }

        if( ! $store->getIsActive()) {
            throw new InactiveStoreException();
        }

        $layout = Mage::getSingleton("core/layout");
        $namespace = $this->getSessionNamespace();

        $layout->setArea(self::GRAPHQL_AREA);

        // Boot session, attempt to restart it if it is invalid.
        //
        // Alternate version of the session-start code in
        // Mage_Core_Controller_Varien_Action::preDispatch(), which had the issue
        // of failing to restart the session properly and instead causing errors
        // to become redirects which is incompatible with an API.
        try {
            $session = Mage::getSingleton("core/session", ["name" => $namespace])
                ->start();
        }
        catch(Mage_Core_Model_Session_Exception $_e) {
            if(PHP_SAPI !== "cli") {
                // The session is invalid, destroy if it is active
                if(session_status() == PHP_SESSION_ACTIVE) {
                    session_destroy();
                    session_commit();
                }

                // New id to reinitialize
                session_regenerate_id(true);
            }

            // Try to reinitialize session
            try {
                // TODO: Do we try to reset the magento instance of the session?
                $session = Mage::getSingleton("core/session", ["name" => $namespace])
                    ->start()
                    // Core is from Mage_Core_Model_Session::__construct()
                    // init is called to make sure the session is properly initialized
                    ->init("core", $namespace);
            }
            catch(Exception $e) {
                throw new SessionFailedException($e);
            }
        }

        // Code from Mage_Core_Controller_Varien_Action::preDispatch() again
        $app->loadArea($layout->getArea());

        return $session;
    }

    public function handleError(array $errors, callable $formatter) {
        // Log all non client-safe exceptions
        foreach($errors as $e) {
            if( ! $e->isClientSafe()) {
                $prev = $e->getPrevious() ?: $e;

                if( ! $prev instanceof Exception) {
                    Mage::logException(new Exception($prev->getMessage(), $prev->getCode(), $prev));
                }
                else {
                    Mage::logException($prev);
                }
            }
        }

        // Default implementation
        return array_map($formatter, $errors);
    }

    public function getResponseCode($result): int {
        $error = 400;

        foreach(is_array($result) ? $result : [$result] as $r) {
            if($r->data !== null || empty($r->errors)) {
                return 200;
            }
            else if( ! empty($r->errors)) {
                foreach($r->errors as $e) {
                    if($e->getCategory() === Error::CATEGORY_INTERNAL) {
                        $error = 500;
                    }
                }
            }
        }

        return $error;
    }

    /**
     * Implmentation of the GraphQL\Server\Helper::parseHttpRequest() for Magento.
     */
    public function parseHttpRequest(Zend_Controller_Request_Http $request, Helper $helper) {
        $method = $request->getMethod();
        $bodyParams = [];
        $urlParams = $request->getQuery();

        if($method === "POST") {
            $contentType = $request->getHeader("Content-Type");

            if($contentType === null) {
                throw new RequestError('Missing "Content-Type" header');
            }

            if(stripos($contentType, "application/graphql") !== false) {
                $bodyParams = ['query' => $request->getRawBody() ?: ''];
            }
            elseif(stripos($contentType, "application/json") !== false) {
                $bodyParams = json_decode($request->getRawBody() ?: "", true);

                if(json_last_error()) {
                    throw new RequestError("Could not parse JSON: " . json_last_error_msg());
                }

                if( ! is_array($bodyParams)) {
                    throw new RequestError(
                        "GraphQL Server expects JSON object or array, but got " .
                        Utils::printSafeJson($bodyParams)
                    );
                }
            }
            elseif(stripos($contentType, "application/x-www-form-urlencoded") !== false ||
              stripos($contentType, "multipart/form-data") !== false) {
                $bodyParams = $request->getPost();
            }
            else {
                throw new RequestError("Unexpected content type: " . Utils::printSafeJson($contentType));
            }
        }

        return $helper->parseRequestParams($method, $bodyParams, $urlParams);
    }

    public function loadSchema(MageQL_Core_Model_Context $context, string $schemaName, bool $includeUnreachable): Schema {
        $config   = $context->getConfig(sprintf("mageql/schema/%s", $schemaName));
        $registry = new Registry();

        $registry->setIncludeUnreachable($includeUnreachable);

        if( ! $config) {
            throw new MageQL_Core_Exception_SchemaNotFound($schemaName);
        }

        if($schemaName === self::SCHEMA_DEFAULT) {
            $registry->addFactory(new DefaultSchema());
        }

        // TODO: Do we reverse this so we can have priority?
        foreach($config as $k => $v) {
            if(array_key_exists("model", $v)) {
                $model = Mage::getModel($v["model"], array_merge([
                    "name" => $k,
                ], $v));

                if(Mage::getIsDeveloperMode() && ! $model instanceof MageQL_Core_Model_Schema_Abstract) {
                    throw new Exception(sprintf("%s: Model instantiated from config '%s' must be an instance of %s", __METHOD__, sprintf("config/default/mageql/schema/%s/%s/model", $schemaName, $k), MageQL_Core_Model_Schema_Abstract::class));
                }

                $model->setContext($context);

                $registry->addFactory($model);

                continue;
            }

            throw new Exception(sprintf("%s: Missing model tag in 'mageql/schema/%s/%s'.", __METHOD__, $schemaName, $k));
        }

        // In case we get warnings we need to keep track of them
        $warnings = [];

        if(Mage::getIsDeveloperMode()) {
            Warning::setWarningHandler(function($msg, $id) use(&$warnings) {
                $warnings[] = [
                    "id"  => $id,
                    "msg" => $msg,
                ];
            });
        }

        $schema = $registry->createSchema();

        if( ! empty($warnings)) {
            // Assert the schema immediately, probably why we got a warning
            $schema->assertValid();

            throw new Exception(sprintf("%s: Warnings: %s", __METHOD__, implode(", ", array_map(function($warning) {
                return sprintf("%s: %s", $warning["id"], $warning["msg"]);
            }, $warnings))));
        }

        if(Mage::getIsDeveloperMode()) {
            Warning::setWarningHandler(null);
        }

        return $schema;
    }
}
