<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Type\AbstractBuilder;
use MageQL\Registry;

class MageQL_Core_Model_Schema_Default extends MageQL_Core_Model_Schema_Abstract {
    public static function resolveRoute($src, array $args, $ctx, ResolveInfo $info) {
        $store = $ctx->getStore();
        $rewrite = Mage::getModel("core/url_rewrite")
            ->setStoreId($store->getId())
            ->loadByRequestPath($args["path"]);

        if($rewrite->getId()) {
            return $rewrite;
        }

        // We also try to load a CMS page if we cannot find anything in the URL rewrite
        $page = Mage::getModel("cms/page");

        $page->setStoreId($ctx->getStore()->getId());
        $page->load($args["path"], "identifier");

        if($page->getId() && $page->getIsActive()) {
            return $page;
        }

        return null;
    }

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "Address":
            return $this->interface("Base address properties");

        case "AddressRegion":
            return $this->object("An address region")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "AddressValidationError":
            return $this->enum("Enum representing possible validation errors for an address", [
                MageQL_Core_Model_Address::MISSING_CITY => [
                    "description" => "City is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_COUNTRY => [
                    "description" => "Country is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_FIRSTNAME => [
                    "description" => "Firstname is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_LASTNAME => [
                    "description" => "Lastname is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_POSTCODE => [
                    "description" => "Postcode is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_REGION => [
                    "description" => "Region is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_STREET => [
                    "description" => "Street is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_TELEPHONE => [
                    "description" => "Telephone is missing a value",
                ],
            ]);

        case "Country":
            return $this->object("A country in an address")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "Customer":
            return $this->object("Type containing customer information")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "LoginCustomerResult":
            return $this->object("Type containing the result of a login operation");

        case "LoginCustomerResultType":
            return $this->enum("Type indicating the result of a login operation", [
                "success" => [
                    "description" => "A successful login",
                ],
                "invalidEmailOrPassword" => [
                    "description" => "Invalid email or password.",
                ],
                "notConfirmed" => [
                    "description" => "The account is not yet confirmed by email.",
                ],
            ]);

        case "Price":
            return $this->interface("Price information which carries tax information");

        case "RouteType":
            return $this->enum("Type indicating variant of route resource", [
                "category" => [
                    "description" => "A category",
                ],
                "cms_page" => [
                    "description" => "A CMS page",
                ],
                "product" => [
                    "description" => "A product",
                ],
                "redirect" => [
                    "description" => "A redirect to another path",
                ],
            ]);

        case "RouteRedirect":
            return $this->object("A response containing a category")
                ->setInterfaces(["Route"]);

        case "Route":
            return $this->interface("Response from a route")
                ->setResolveType(function($rewrite) use($registry) {
                    if($rewrite->getUrlRewriteId()) {
                        $isRedirect = stripos($rewrite->getOptions() ?: "", "R") !== false;

                        if($isRedirect) {
                            return $registry->getType("RouteRedirect");
                        }

                        // We have to check if it is a product first, since it
                        // might be a product inside of a category
                        if($rewrite->getProductId()) {
                            return $registry->getType("RouteProduct");
                        }

                        if($rewrite->getCategoryId()) {
                            return $registry->getType("RouteCategory");
                        }
                    }
                    else if($rewrite->getPageId()) {
                        return $registry->getType("RouteCmsPage");
                    }

                    throw new Exception("Unknown route type");
                });

        case "StoreInfo":
            return $this->object("Basic store information")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "UpdateCustomerEmailResult":
            return $this->object("Result of updateCustomerEmail mutation");

        case "UpdateCustomerEmailResultType":
            return $this->enum("Type of result from updateCustomerEmail mutation", [
                MageQL_Core_Model_Customer::SUCCESS => [
                    "description" => "Email was successfully update",
                ],
                MageQL_Core_Model_Customer::NOT_MODIFIED => [
                    "description" => "Email was successfully update",
                ],
                MageQL_Core_Model_Customer::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in",
                ],
                MageQL_Core_Model_Customer::ERROR_INVALID_EMAIL_ADDRESS => [
                    "description" => "The supplied email address is invalid.",
                ],
                MageQL_Core_Model_Customer::ERROR_EMAIL_EXISTS => [
                    "description" => "The supplied email is used by another customer",
                ],
            ]);
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "Address":
            return [
                "prefix" => $this->field("String", "Name prefix"),
                "firstname" => $this->field("String", "Firstname"),
                "middlename" => $this->field("String", "Name prefix"),
                "lastname" => $this->field("String", "Lastname"),
                "suffix" => $this->field("String", "Name suffix"),
                "company" => $this->field("String", "Company"),
                "street" => $this->field("[String!]!", "Street")
                    ->setResolver("MageQL_Core_Model_Address::resolveStreet"),
                "city" => $this->field("String", "City"),
                "postcode" => $this->field("String", "Postcode"),
                "region" => $this->field("AddressRegion", "Region")
                    ->setResolver("MageQL_Core_Model_Address::resolveRegion"),
                "country" => $this->field("Country!", "Country")
                    ->setResolver("MageQL_Core_Model_Address::resolveCountry"),
                "telephone" => $this->field("String", "Telephone"),
            ];

        case "AddressRegion":
            return [
                "code" => $this->field("String!", "Region code, ISO 3166-2"),
                "name" => $this->field("String!", "Region name"),
            ];

        case "Country":
            return [
                "code" => $this->field("ID!", "Country code, ISO 3166-2")
                   ->setResolver("MageQL_Core_Model_Address::resolveCountryCode"),
               "name" => $this->field("String!", "Country name for the current locale"),
            ];

        case "Customer":
            // Name, description and everything:
            $config = Mage::getSingleton("mageql/attributes_customer");
            // Category index everything, including all attribute sets
            $attrs = $config->getAttributes();
            $fields = $config->createFields($attrs);

            // TODO: Customer group
            return array_merge($fields, [
                "email" => $this->field("String!", "Customer email"),
                "createdAt" => $this->field("String!", "Customer created date")
                    ->setResolver("MageQL_Core_Model_Customer::resolveCreatedAt"),
            ]);

        case "LoginCustomerResult":
            return [
                "result" => $this->field("LoginCustomerResultType!", "The result type")
                    ->setResolver(function($src) {
                        // Login is successful if we have an object here
                        return is_string($src) ? $src : "success";
                    }),
                "customer" => $this->field("Customer", "Present if it is a success")
                    ->setResolver(function($src) {
                        // Login is successful if we have an object here
                        return is_string($src) ? null : $src;
                    }),
            ];

        case "Mutation":
            return [
                // TODO: More customer mutations
                "loginCustomer" => $this->field("LoginCustomerResult!", "Attempts to log in a customer")
                    ->addArgument("email", $this->argument("String!", "The email of the customer account"))
                    ->addArgument("password", $this->argument("String!", "The password"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateLogin"),
                "logoutCustomer" => $this->field("Boolean!", "Attempts to log out a customer")
                    ->setResolver("MageQL_Core_Model_Customer::mutateLogout"),
                "updateCustomerEmail" => $this->field("UpdateCustomerEmailResult!", "Updates the email for the currently logged in customer, this will also propagate to any quote.")
                    ->addArgument("email", $this->argument("String!", "The new email for the account"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateUpdateEmail"),
            ];

        case "Price":
            return [
                "exVat" => $this->field("Float!", "Price excluding VAT"),
                "incVat" => $this->field("Float!", "Price including VAT"),
                "vat" => $this->field("Float!", "VAT amount"),
            ];

        case "Query":
            return [
                "customer" => $this->field("Customer", "Customer information if a customer is logged in")
                    ->setResolver("MageQL_Core_Model_Customer::resolveCustomer"),
                "info" => $this->field("StoreInfo!", "Information about the current store")
                    ->setResolver("MageQL\\forwardResolver"),
                "route" => $this->field("Route", "Attempt to fetch a resource by its route")
                    ->addArgument("path", $this->argument("String!", "Path to resource"))
                    ->setResolver(__CLASS__."::resolveRoute"),
            ];

        case "RouteRedirect":
            return [
                "isPermanent" => $this->field("Boolean!", "If the redirect is a permanent redirect")
                    ->setResolver(function($src) {
                        return stripos($src->getOptions(), "P") !== false;
                    }),
                "type" => $this->field("RouteType!", "Type of route")
                    ->setResolver(function() {
                        return "redirect";
                    }),
                "url" => $this->field("String!", "Redirect to")
                    ->setResolver(function($src) {
                        $url = $src->getTargetPath();

                        // Allow external redirects
                        if(strpos($url, "https://") === 0 || strpos($url, "http://") === 0) {
                            return $url;
                        }

                        // Always provide the root
                        return "/".$url;
                    }),
            ];

        case "Route":
            return [
                "type" => $this->field("RouteType!", "Type of route resource")
                    ->setResolver(__CLASS__."::resolveRouteType"),
            ];

        case "StoreInfo":
            return [
                "baseCurrencyCode" => $this->field("String!", "Base currency code"),
                "baseUrl" => $this->field("String!", "Base URL"),
                "currencyCodes" => $this->field("[String!]!", "Currency codes")
                    ->setResolver("MageQL_Core_Model_Store::resolveCurrencyCodes"),
                "countries" => $this->field("[Country!]!", "List of available countries for the store")
                    ->setResolver("MageQL_Core_Model_Store::resolveCountries"),
                "defaultCountry" => $this->field("Country!", "Default country for the store")
                    ->setResolver("MageQL_Core_Model_Store::resolveDefaultCountry"),
                "defaultDescription" => $this->field("String", "Default page description")
                    ->setResolver("MageQL_Core_Model_Store::resolveDefaultDescription"),
                "defaultTitle" => $this->field("String", "Default page title")
                    ->setResolver("MageQL_Core_Model_Store::resolveDefaultTitle"),
                "locale" => $this->field("String!", "Locale code for the store")
                    ->setResolver("MageQL_Core_Model_Store::resolveLocale"),
                "name" => $this->field("String!", "Name"),
                "titlePrefix" => $this->field("String", "Page title prefix")
                    ->setResolver("MageQL_Core_Model_Store::resolveTitlePrefix"),
                "titleSuffix" => $this->field("String", "Page title suffix")
                    ->setResolver("MageQL_Core_Model_Store::resolveTitleSuffix"),
            ];

        case "UpdateCustomerEmailResult":
            return [
                "result" => $this->field("UpdateCustomerEmailResultType!", "The type of result")
                    ->setResolver("MageQL_Core_Model_Customer_Result::resolveResult"),
                "customer" => $this->field("Customer", "The customer result, if available")
                    ->setResolver("MageQL_Core_Model_Customer_Result::resolveCustomer"),
            ];
        }

        return [];
    }

    public function getUnreachableTypes(): array {
        return [
            "Address",
            "AddressValidationError",
            "Price",
            "RouteRedirect",
        ];
    }
}
