<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Core_Model_Customer extends Mage_Core_Model_Abstract {
    const SUCCESS = "success";
    const NOT_MODIFIED = "notModified";
    const ERROR_NOT_LOGGED_IN = "errorNotLoggedIn";
    const ERROR_INVALID_EMAIL_ADDRESS = "errorInvalidEmail";
    const ERROR_EMAIL_EXISTS = "errorEmailExists";

    public static function resolveCustomer($src, array $args, $ctx, ResolveInfo $info) {
        $session = Mage::getSingleton("customer/session");

        if($session->isLoggedIn()) {
            return $session->getCustomer();
        }

        return null;
    }

    public static function resolveCreatedAt(Mage_Customer_Model_Customer $src) {
        return gmdate("Y-m-d\TH:i:s\Z", strtotime($src->getCreatedAt()));
    }

    public static function mutateLogin($src, array $args, $ctx, ResolveInfo $info) {
        $email = $args["email"];
        $password = $args["password"];
        try {
            // TODO: Should these be a part of the context?
            $session = Mage::getSingleton("customer/session");
            // TODO: Attributes?
            $customer = Mage::getModel("customer/customer")
                ->setStore($ctx->getStore())
                ->loadByEmail($email);

            $session->login($email, $password);
            $session->setCustomerAsLoggedIn($customer);
            $customer->setSession($session->getSessionId());

            return $customer;
        }
        catch(Mage_Core_Exception $e) {
            if($e->getMessage() === Mage::helper("customer")->__("Invalid login or password.")) {
                return "invalidEmailOrPassword";
            }

            if($e->getMessage() === Mage::helper("customer")->__("This account is not confirmed.")) {
                return "notConfirmed";
            }

            throw $e;
        }
    }

    public static function mutateLogout() {
        $session = Mage::getSingleton("customer/session");

        if($session->isLoggedIn()) {
            $session->logout();

            return true;
        }

        return false;
    }

    public static function mutateUpdateEmail($src, array $args) {
        $session = Mage::getSingleton("customer/session");
        $email = trim($args["email"]);

        if( ! $session->isLoggedIn()) {
            return new MageQL_Core_Model_Customer_Result(self::ERROR_NOT_LOGGED_IN);
        }

        $customer = $session->getCustomer();

        if( ! Zend_Validate::is($email, "EmailAddress")) {
            return new MageQL_Core_Model_Customer_Result(self::ERROR_INVALID_EMAIL_ADDRESS, $customer);
        }

        if($email === $customer->getEmail()) {
            return new MageQL_Core_Model_Customer_Result(self::NOT_MODIFIED, $customer);
        }

        // Save the old email so we can notify customer
        $customer->setOldEmail($customer->getEmail());
        $customer->setEmail($email);
        $customer->setRpToken(null);
        $customer->setRpTokenCreatedAt(null);

        try {
            $customer->save();
        }
        catch(Mage_Customer_Exception $e) {
            if($e->getCode() === Mage_Customer_Model_Customer::EXCEPTION_EMAIL_EXISTS) {
                return new MageQL_Core_Model_Customer_Result(self::ERROR_EMAIL_EXISTS, $customer);
            }

            throw $e;
        }

        $customer->sendChangedPasswordOrEmail();

        // Ensure we update the email in the quote if any
        $checkout = Mage::getSingleton("checkout/session");

        if($checkout->hasQuote()) {
            $quote = $checkout->getQuote();

            $quote->setCustomer($customer);
            $quote->save();
        }

        return new MageQL_Core_Model_Customer_Result(self::SUCCESS, $customer);
    }
}
