<?php

declare(strict_types=1);

namespace MageQL;

use Exception;
use Varien_Object;
use Mage_Core_Model_Abstract;

use GraphQL\Type\Definition\ResolveInfo;

/**
 * Forwards the source to the next resolver.
 */
function forwardResolver($a) {
    return $a;
}

function snakeCaseToCamel($str) {
    // TODO: Cache?
    return lcfirst(str_replace("_", "", ucwords($str, "_")));
}

function spacedToCamel($str) {
    return str_replace([" ", "-"], ["", "_"], $str);
}

function camelToSnakeCase($str) {
    // TODO: Can we reuse the Varien_Object cache somehow?
    return strtolower(preg_replace("/(.)([A-Z])/", "$1_$2", $str));
}

function defaultVarienObjectResolver(Varien_Object $src, $args, $context, ResolveInfo $info) {
    $getter = "get".ucfirst($info->fieldName);
    $has = "has".ucfirst($info->fieldName);

    if(method_exists($src, $getter) || $src->$has()) {
        return $src->$getter();
    }

    return null;
}

function dateAttributeResolver(Varien_Object $src, $args, $context, ResolveInfo $info) {
    $value = defaultVarienObjectResolver($src, $args, $context, $info);

    if($value !== null) {
        $parsed = strtotime($value);

        return gmdate("Y-m-d\TH:i:s\Z", $parsed);
    }

    return null;
}

function attributeSourceModel($src, $name) {
    $attributeCode = camelToSnakeCase($name);

    // Essentially copied from Mage_Catalog_Model_Product::getAttributeText
    // This is missing on Categories and Customer, and here modified to also support
    // multiselect with array-storage
    $attrResource = $src->getResource()->getAttribute($attributeCode);

    // If the attribute no longer exists
    if( ! $attrResource) {
        $error = sprintf(
            "%s: Could not load attribute resource model for '%s' of entity %s in store %d.",
            __FUNCTION__,
            $attributeCode,
            get_class($src),
            Mage::app()->getStore()->getId()
        );

        if(Mage::getIsDeveloperMode()) {
            throw new Exception($error);
        }
        else {
            Mage::log($error, Zend_Log::ERR);
        }

        return null;
    }

    return $attrResource->getSource();
}

function selectAttributeResolver(Mage_Core_Model_Abstract $src, $args, $context, ResolveInfo $info) {
    $value = defaultVarienObjectResolver($src, $args, $context, $info);
    $model = attributeSourceModel($src, $info->fieldName);

    if( ! $model) {
        return null;
    }

    return $model->getOptionText($value) ?: $value;
}

function multiselectAttributeResolver(Mage_Core_Model_Abstract $src, $args, $context, ResolveInfo $info) {
    $value = defaultVarienObjectResolver($src, $args, $context, $info);
    $model = attributeSourceModel($src, $info->fieldName);

    if( ! $model) {
        return null;
    }

    return array_filter(array_map([$model, "getOptionText"], array_map("trim", is_array($value) ? $value : explode(",", $value))));
}
