<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Schema\AbstractSchema;
use MageQL\Type\AbstractBuilder;
use MageQL\Registry;

class MageQL_Core_Model_Schema_Default extends AbstractSchema {
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "Address":
            return $this->interface("Base address properties");

        case "AddressRegion":
            return $this->object("An address region")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "AddressValidationError":
            return $this->enum("Enum representing possible validation errors for an address", [
                MageQL_Core_Model_Address::MISSING_CITY => [
                    "description" => "City is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_COUNTRY => [
                    "description" => "Country is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_FIRSTNAME => [
                    "description" => "Firstname is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_LASTNAME => [
                    "description" => "Lastname is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_POSTCODE => [
                    "description" => "Postcode is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_REGION => [
                    "description" => "Region is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_STREET => [
                    "description" => "Street is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_TELEPHONE => [
                    "description" => "Telephone is missing a value",
                ],
            ]);

        case "Country":
            return $this->object("A country in an address")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "ImageData":
            return $this->object("Image data");

        case "Price":
            return $this->interface("Price information which carries tax information");

        case "RouteType":
            return $this->enum("Type indicating variant of route resource", [
                "category" => [
                    "description" => "A category",
                ],
                "cms_page" => [
                    "description" => "A CMS page",
                ],
                "product" => [
                    "description" => "A product",
                ],
                "redirect" => [
                    "description" => "A redirect to another path",
                ],
            ]);

        case "RouteRedirect":
            return $this->object("A response containing a category")
                ->setInterfaces(["Route"]);

        case "Route":
            return $this->interface("Response from a route")
                ->setTypeResolver("MageQL_Core_Model_Route::typeResolver");

        case "StoreInfo":
            return $this->object("Basic store information")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "Address":
            $config = Mage::getSingleton("mageql/attributes_customer_address");

            return $config->createFields($config->getAttributes());

        case "AddressRegion":
            return [
                "code" => $this->field("String!", "Region code, ISO 3166-2"),
                "name" => $this->field("String!", "Region name"),
            ];

        case "Country":
            return [
                "code" => $this->field("ID!", "Country code, ISO 3166-2")
                   ->setResolver("MageQL_Core_Model_Address::resolveCountryCode"),
                "name" => $this->field("String!", "Country name for the current locale"),
            ];

        case "ImageData":
            return [
                "src" => $this->field("String!", "The image URL")
                    ->addArgument("width", $this->argument("Int", "Maximum width of the image"))
                    ->addArgument("height", $this->argument("Int", "Minimum width of the image"))
                    ->addArgument(
                        "fill",
                        $this->argument("Boolean", "If to fill the image to the given size")
                             ->setDefaultValue(false)
                    )->setResolver("MageQL_Core_Model_Attributes_Image_Abstract::resolveSrc"),
            ];

        case "Price":
            return [
                "exVat" => $this->field("Float!", "Price excluding VAT")
                    ->setResolver("MageQL_Core_Model_Price_Abstract::resolveExVat"),
                "incVat" => $this->field("Float!", "Price including VAT")
                    ->setResolver("MageQL_Core_Model_Price_Abstract::resolveIncVat"),
                "vat" => $this->field("Float!", "VAT amount")
                    ->setResolver("MageQL_Core_Model_Price_Abstract::resolveVat"),
            ];

        case "Query":
            return [
                "info" => $this->field("StoreInfo!", "Information about the current store")
                    ->setResolver("MageQL\\forwardResolver"),
                "route" => $this->field("Route", "Attempt to fetch a resource by its route")
                    ->addArgument("path", $this->argument("String!", "Path to resource"))
                    ->setResolver("MageQL_Core_Model_Route::resolve"),
            ];

        case "RouteRedirect":
            return [
                "isPermanent" => $this->field("Boolean!", "If the redirect is a permanent redirect")
                    ->setResolver("MageQL_Core_Model_Route_Redirect::resolveIsPermanent"),
                "type" => $this->field("RouteType!", "Type of route")
                    ->setResolver("MageQL_Core_Model_Route_Redirect::resolveType"),
                "url" => $this->field("String!", "Redirect to")
                    ->setResolver("MageQL_Core_Model_Route_Redirect::resolveUrl"),
            ];

        case "Route":
            return [
                "type" => $this->field("RouteType!", "Type of route resource"),
            ];

        case "StoreInfo":
            return [
                "baseCurrencyCode" => $this->field("String!", "Base currency code"),
                "baseUrl" => $this->field("String!", "Base URL"),
                "currencyCodes" => $this->field("[String!]!", "Currency codes")
                    ->setResolver("MageQL_Core_Model_Store::resolveCurrencyCodes"),
                "countries" => $this->field("[Country!]!", "List of available countries for the store")
                    ->setResolver("MageQL_Core_Model_Store::resolveCountries"),
                "defaultCountry" => $this->field("Country!", "Default country for the store")
                    ->setResolver("MageQL_Core_Model_Store::resolveDefaultCountry"),
                "defaultDescription" => $this->field("String", "Default page description")
                    ->setResolver("MageQL_Core_Model_Store::resolveDefaultDescription"),
                "defaultTitle" => $this->field("String", "Default page title")
                    ->setResolver("MageQL_Core_Model_Store::resolveDefaultTitle"),
                "locale" => $this->field("String!", "Locale code for the store")
                    ->setResolver("MageQL_Core_Model_Store::resolveLocale"),
                "name" => $this->field("String!", "Name"),
                "titlePrefix" => $this->field("String", "Page title prefix")
                    ->setResolver("MageQL_Core_Model_Store::resolveTitlePrefix"),
                "titleSuffix" => $this->field("String", "Page title suffix")
                    ->setResolver("MageQL_Core_Model_Store::resolveTitleSuffix"),
            ];
        }

        return [];
    }

    public function getUnreachableTypes(): array {
        return [
            "Price",
            "RouteRedirect",
        ];
    }
}
