<?php

declare(strict_types=1);

namespace MageQL\Core;

use Mage;
use Throwable;
use Varien_Profiler;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

require_once __DIR__."/RequestTest/Schema.php";

class RequestTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/mageql/schema/testing/TheTest");
    }

    public function testMissingGetParameters(): void {
        // Test without developer mode for fails to avoid a stack trace
        Mage::setIsDeveloperMode(false);

        $resp = MagentoManager::runRequest(new Request("GET /graphql", []));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals("", MagentoManager::readLogFile());
    }

    public function testEmptyPostRequest(): void {
        Mage::setIsDeveloperMode(false);
        $resp = MagentoManager::runRequest(new Request("POST /graphql", []));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertStringContainsString("Missing \\\"Content-Type\\\" header", MagentoManager::readLogFile());
        $this->assertStringContainsString("client_errors", MagentoManager::readLogFile());
    }

    public function testEmptyQueryGraphQL(): void {
        Mage::setIsDeveloperMode(false);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], ""));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals("", MagentoManager::readLogFile());
    }

    public function testGrpahQLQuery(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], "query { info { baseUrl } }"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testJSONQuery(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], '{"query":"query { info { baseUrl } }"}'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testMultiQuerySingle(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], '[{"query":"query { info { baseUrl } }"}]'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testMultiQuery(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], '[{"query":"query { info { baseUrl } }"}, {"query": "query { info { name } }"}]'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testMultiQueryError(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], '[{"query":"query { info { baseUrl } }"}, {"query": "query { doesNotExist { name } }"}]'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testNoGraphQL(): void {
        $resp = MagentoManager::runRequest(new Request("GET /", []));

        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertNotEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testMissingType(): void {
        Mage::setIsDeveloperMode(false);
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], '{
        "query": "query ($foo: MissingTypeHere) { info { baseUrl } }",
        "variables": "{\"foo\":{}}"
    }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testInitEvent(): void {
        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'query { info { baseUrl } }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_context_after", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testInitEvent2(): void {
        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'this is not a graphql request'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(400, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_context_after", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testOtherSchema(): void {
        // Test without developer mode for fails to avoid a stack trace
        Mage::setIsDeveloperMode(false);

        $resp = MagentoManager::runRequest(new Request("POST /graphql/testing", [
            "Content-Type" => "application/graphql",
        ], "query { info { baseUrl } }"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertStringContainsString("MageQL (1001): Schema 'testing' was not found", MagentoManager::readLogFile());
        $this->assertStringContainsString("client_errors", MagentoManager::readLogFile());
    }

    public function testOtherSchema2(): void {
        Config::setConfigPath("global/models/custom/class", "RequestTestModel");
        Config::setConfigPath("default/mageql/schema/testing/TheTest/model", "custom/schema");

        $resp = MagentoManager::runRequest(new Request("POST /graphql/testing", [
            "Content-Type" => "application/graphql",
        ], "query { testIt }"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertEquals("", MagentoManager::readLogFile());
    }
}
