<?php

declare(strict_types=1);

use GraphQL\Error\Error;
use GraphQL\Executor\Values;
use GraphQL\Language\AST\FieldNode;
use GraphQL\Language\AST\FragmentSpreadNode;
use GraphQL\Language\AST\InlineFragmentNode;
use GraphQL\Language\AST\Node;
use GraphQL\Language\AST\NodeKind;
use GraphQL\Language\AST\OperationDefinitionNode;
use GraphQL\Language\AST\SelectionSetNode;
use GraphQL\Language\Visitor;
use GraphQL\Language\VisitorOperation;
use GraphQL\Language\Printer;
use GraphQL\Type\Definition\Directive;
use GraphQL\Type\Definition\FieldDefinition;
use GraphQL\Validator\Rules\QueryComplexity;
use GraphQL\Validator\ValidationContext;

/**
 * This class is a copy from GraphQL\Validator\Rules\QueryComplexity with some additions
 */
class MageQL_Core_Model_Validation_QueryComplexity extends QueryComplexity
{
    /** @var int */
    private $maxQueryComplexity;

    /** @var int */
    private $warnQueryComplexity;

    /** @var ArrayObject */
    private $variableDefs;

    /** @var ArrayObject */
    private $fieldNodeAndDefs;

    /** @var ValidationContext */
    private $context;

    /** @var int */
    private $complexity;

    /**
     * @param int $maxQueryComplexity
     * @param int $warnQueryComplexity
     */
    public function __construct($maxQueryComplexity = 0, $warnQueryComplexity = 0)
    {
        $this->setMaxQueryComplexity($maxQueryComplexity);
        $this->setWarnQueryComplexity($warnQueryComplexity);
    }

    public function getVisitor(ValidationContext $context)
    {
        $this->context = $context;

        $this->variableDefs     = new ArrayObject();
        $this->fieldNodeAndDefs = new ArrayObject();
        $this->complexity       = 0;

        return $this->invokeIfNeeded(
            $context,
            [
                NodeKind::SELECTION_SET        => function (SelectionSetNode $selectionSet) use ($context) : void {
                    $this->fieldNodeAndDefs = $this->collectFieldASTsAndDefs(
                        $context,
                        $context->getParentType(),
                        $selectionSet,
                        null,
                        $this->fieldNodeAndDefs
                    );
                },
                NodeKind::VARIABLE_DEFINITION  => function ($def) : VisitorOperation {
                    $this->variableDefs[] = $def;

                    return Visitor::skipNode();
                },
                NodeKind::OPERATION_DEFINITION => [
                    'leave' => function (OperationDefinitionNode $operationDefinition) use ($context, &$complexity) : void {
                        $errors = $context->getErrors();

                        if (count($errors) > 0) {
                            return;
                        }

                        $complexity = $this->fieldComplexity($operationDefinition, $complexity);

                        if (
                            $this->getWarnQueryComplexity() !== self::DISABLED
                            && $complexity >= $this->getWarnQueryComplexity()
                        ) {
                            Mage::log(
                                sprintf('[MageQL] Query complexity: %d', $complexity),
                                Zend_Log::WARN,
                                MageQL_Core_Helper_Data::LOG_CHANNEL
                            );
                            // Do not print graphql query that exceeded threshold
                            /*Mage::log(
                                Printer::doPrint($operationDefinition),
                                Zend_Log::WARN,
                                MageQL_Core_Helper_Data::LOG_CHANNEL
                            );*/
                        }

                        if (
                            $this->getMaxQueryComplexity() !== self::DISABLED
                            && $complexity >= $this->getMaxQueryComplexity()
                        ) {
                            $context->reportError(
                                new Error(self::maxQueryComplexityErrorMessage(
                                    $this->getMaxQueryComplexity(),
                                    $complexity
                                ))
                            );
                        }
                    },
                ],
            ]
        );
    }

    /**
     * @param Node $node
     * @param int $complexity
     * @return int
     */
    private function fieldComplexity($node, $complexity = 0)
    {
        if (isset($node->selectionSet) && $node->selectionSet instanceof SelectionSetNode) {
            foreach ($node->selectionSet->selections as $childNode) {
                $complexity = $this->nodeComplexity($childNode, $complexity);
            }
        }

        return $complexity;
    }

    /**
     * @param Node $node
     * @param int $complexity
     * @return int
     */
    private function nodeComplexity(Node $node, $complexity = 0)
    {
        switch (true) {
            case $node instanceof FieldNode:
                // default values
                $args         = [];
                $complexityFn = FieldDefinition::DEFAULT_COMPLEXITY_FN;

                // calculate children complexity if needed
                $childrenComplexity = 0;

                // node has children?
                if (isset($node->selectionSet)) {
                    $childrenComplexity = $this->fieldComplexity($node);
                }

                $astFieldInfo = $this->astFieldInfo($node);
                $fieldDef     = $astFieldInfo[1];

                if ($fieldDef instanceof FieldDefinition) {
                    if ($this->directiveExcludesField($node)) {
                        break;
                    }

                    $args = $this->buildFieldArguments($node);
                    //get complexity fn using fieldDef complexity
                    if (method_exists($fieldDef, 'getComplexityFn')) {
                        $complexityFn = $fieldDef->getComplexityFn();
                    }
                }

                $complexity += $complexityFn($childrenComplexity, $args);
                break;

            case $node instanceof InlineFragmentNode:
                // node has children?
                if (isset($node->selectionSet)) {
                    $complexity = $this->fieldComplexity($node, $complexity);
                }
                break;

            case $node instanceof FragmentSpreadNode:
                $fragment = $this->getFragment($node);

                if ($fragment !== null) {
                    $complexity = $this->fieldComplexity($fragment, $complexity);
                }
                break;
        }

        return $complexity;
    }

    /**
     * @return array
     */
    private function astFieldInfo(FieldNode $field)
    {
        $fieldName    = $this->getFieldName($field);
        $astFieldInfo = [null, null];
        if (isset($this->fieldNodeAndDefs[$fieldName])) {
            foreach ($this->fieldNodeAndDefs[$fieldName] as $astAndDef) {
                if ($astAndDef[0] === $field) {
                    $astFieldInfo = $astAndDef;
                    break;
                }
            }
        }

        return $astFieldInfo;
    }

    /**
     * @return bool
     */
    private function directiveExcludesField(FieldNode $node)
    {
        foreach ($node->directives as $directiveNode) {
            if ($directiveNode->name->value === 'deprecated') {
                return false;
            }
            [$errors, $variableValues] = Values::getVariableValues(
                $this->context->getSchema(),
                $this->variableDefs,
                $this->getRawVariableValues()
            );
            if (count($errors ?? []) > 0) {
                throw new Error(implode(
                    "\n\n",
                    array_map(
                        static function ($error) {
                            return $error->getMessage();
                        },
                        $errors
                    )
                ));
            }
            if ($directiveNode->name->value === 'include') {
                $directive = Directive::includeDirective();
                /** @var bool $directiveArgsIf */
                $directiveArgsIf = Values::getArgumentValues($directive, $directiveNode, $variableValues)['if'];

                return ! $directiveArgsIf;
            }
            if ($directiveNode->name->value === Directive::SKIP_NAME) {
                $directive = Directive::skipDirective();
                /** @var bool $directiveArgsIf */
                $directiveArgsIf = Values::getArgumentValues($directive, $directiveNode, $variableValues)['if'];

                return $directiveArgsIf;
            }
        }

        return false;
    }

    /**
     * @return array
     */
    private function buildFieldArguments(FieldNode $node)
    {
        $rawVariableValues = $this->getRawVariableValues();
        $astFieldInfo      = $this->astFieldInfo($node);
        $fieldDef          = $astFieldInfo[1];

        $args = [];

        if ($fieldDef instanceof FieldDefinition) {
            [$errors, $variableValues] = Values::getVariableValues(
                $this->context->getSchema(),
                $this->variableDefs,
                $rawVariableValues
            );

            if (count($errors ?? []) > 0) {
                throw new Error(implode(
                    "\n\n",
                    array_map(
                        static function ($error) {
                            return $error->getMessage();
                        },
                        $errors
                    )
                ));
            }

            $args = Values::getArgumentValues($fieldDef, $node, $variableValues);
        }

        return $args;
    }

    public function getWarnQueryComplexity(): int
    {
        return $this->warnQueryComplexity;
    }

    /**
     * Set warn query complexity. Must be greater or equal to 0.
     */
    public function setWarnQueryComplexity(int $warnQueryComplexity): void
    {
        $this->checkIfGreaterOrEqualToZero('warnQueryComplexity', $warnQueryComplexity);

        $this->warnQueryComplexity = (int) $warnQueryComplexity;
    }

    protected function isEnabled(): bool
    {
        return $this->getMaxQueryComplexity() !== self::DISABLED
            || $this->getWarnQueryComplexity() !== self::DISABLED;
    }
}
