<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;
use MageQL\Context;
use MageQL\Type\AbstractBuilder;
use MageQL\Registry;

/**
 * @psalm-suppress PropertyNotSetInConstructor
 */
class MageQL_Core_Model_Route extends Mage_Core_Model_Abstract {
    const EVENT_ROUTE_RESOLVE = "mageql_core_route_resolve";

    /**
     * @param mixed $unusedSrc
     * @param array{path:string} $args
     * @return Mage_Core_Model_Url_Rewrite|Mage_Cms_Model_Page|null
     */
    public static function resolve(
        $unusedSrc,
        array $args,
        Context $ctx,
        ResolveInfo $info
    ) {
        $store = $ctx->getStore();
        $rewrite = Mage::getModel("core/url_rewrite")
            ->setStoreId($store->getId())
            ->loadByRequestPath($args["path"]);

        if($rewrite->getId()) {
            $result = new Varien_Object([
                "rewrite" => $rewrite,
            ]);

            Mage::dispatchEvent(self::EVENT_ROUTE_RESOLVE, [
                "context" => $ctx,
                "info" => $info,
                "is_redirect" => self::isRedirect($rewrite),
                "result" => $result,
                "rewrite" => $rewrite,
            ]);

            return $result->getRewrite();
        }

        // We also try to load a CMS page if we cannot find anything in the URL rewrite
        $page = Mage::getModel("cms/page");
        $pageId = $page->checkIdentifier($args["path"], $ctx->getStore()->getId());

        if( ! $pageId) {
            return null;
        }

        $page->load((int)$pageId);

        if($page->getId() && $page->getIsActive()) {
            return $page;
        }

        return null;
    }

    /**
     * @param Mage_Core_Model_Url_Rewrite|Mage_Cms_Model_Page $rewrite
     */
    public static function isRedirect($rewrite): bool {
        return stripos($rewrite->getData("options") ?: "", "R") !== false;
    }

    /**
     * @param Mage_Core_Model_Url_Rewrite|Mage_Cms_Model_Page $rewrite
     */
    public static function typeResolver($rewrite): string {
        if($rewrite->hasData("url_rewrite_id")) {
            // Redirects could be to products/categories
            if(self::isRedirect($rewrite)) {
                return "RouteRedirect";
            }

            // We have to check if it is a product first, since it
            // might be a product inside of a category
            if($rewrite->getData("product_id")) {
                return "RouteProduct";
            }

            if($rewrite->getData("category_id")) {
                return "RouteCategory";
            }
        }
        else if($rewrite->getData("page_id")) {
            return "RouteCmsPage";
        }

        throw new Exception("Unknown route type");
    }
}
