<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;
use MageQL\Context;

/**
 * @psalm-suppress PropertyNotSetInConstructor
 */
class MageQL_Core_Model_Customer extends Mage_Core_Model_Abstract {
    const NOT_MODIFIED = "notModified";

    // Migration script converts this old config value to new
    // const CONFIG_EMAIL_UPDATE = "customer/changed_account/graphql_email_update_enabled";

    const FIELD_NORMAL = 'normal';
    const FIELD_READONLY = 'readonly';
    const FIELD_HIDDEN = 'hidden';

    const CONFIG_FIELD_EMAIL = "customer/changed_account/graphql_field_email";
    const CONFIG_FIELD_PHONE = "customer/changed_account/graphql_field_phone";
    const CONFIG_FIELD_NAME = "customer/changed_account/graphql_field_name";
    const CONFIG_FIELD_COMPANY = "customer/changed_account/graphql_field_company";
    const CONFIG_FIELD_ADDRESS = "customer/changed_account/graphql_field_address";
    const CONFIG_FIELD_COUNTRY = "customer/changed_account/graphql_field_country";

    // syntax: <field name> => <config path>
    const FIELD_SETTINGS = [
        // email
        "email" => self::CONFIG_FIELD_EMAIL,

        // phone
        "telephone" => self::CONFIG_FIELD_PHONE,

        // name
        "prefix" => self::CONFIG_FIELD_NAME,
        "firstname" => self::CONFIG_FIELD_NAME,
        "middlename" => self::CONFIG_FIELD_NAME,
        "lastname" => self::CONFIG_FIELD_NAME,
        "suffix" => self::CONFIG_FIELD_NAME,

        // company
        "company" => self::CONFIG_FIELD_COMPANY,

        // address
        "address" => self::CONFIG_FIELD_ADDRESS,
        "postcode" => self::CONFIG_FIELD_ADDRESS,
        "city" => self::CONFIG_FIELD_ADDRESS,
        "region" => self::CONFIG_FIELD_ADDRESS,

        // country
        "country" => self::CONFIG_FIELD_COUNTRY,
    ];

    /**
     * Event triggered when a customer was successfully updated by the
     * updateCustomer mutation.
     *
     * Parameters:
     *
     *  * context: Context
     *  * customer: Mage_Customer_Model_Customer
     */
    const EVENT_MUTATION_UPDATE_CUSTOMER_SUCCESS = "mageql_core_mutation_updateCustomer_success";

    /**
     * Event triggered when a customer email was successfully updated by the
     * updateCustomerEmail mutation.
     *
     * Parameters:
     *
     *  * context: Context
     *  * customer: Mage_Customer_Model_Customer
     */
    const EVENT_MUTATION_UPDATE_CUSTOMER_EMAIL_SUCCESS = "mageql_core_mutation_updateCustomerEmail_success";

    /**
     * Event triggered when a customer address default was successfully updated
     * by the setCustomerDefaultAddress mutation.
     *
     * Parameters:
     *
     *  * address: MageQL_Customer_Model_Address
     *  * context: Context
     *  * customer: Mage_Customer_Model_Customer
     *  * type: "default_billing"|"default_shipping"
     */
    const EVENT_MUTATION_SET_CUSTOMER_DEFAULT_ADDRESS_SUCCESS = "mageql_core_mutation_setCustomerDefaultAddress_success";

    public static function resolveCustomer(): ?Mage_Customer_Model_Customer {
        $session = Mage::getSingleton("customer/session");

        if($session->isLoggedIn()) {
            return $session->getCustomer();
        }

        return null;
    }

    public static function resolveCreatedAt(
        Mage_Customer_Model_Customer $src
    ): string {
        return gmdate("Y-m-d\TH:i:s\Z", strtotime($src->getCreatedAt()));
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function mutateLogin(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Login {
        if( ! $ctx->getConfig(MageQL_Core_Helper_Data::CONFIG_CUSTOMER_STARTUP_ENABLE_LOGIN)) {
            return new MageQL_Core_Model_Customer_Result_Login(
                MageQL_Core_Model_Customer_Result_Login::ERROR_LOGIN_NOT_ENABLED
            );
        }

        $email = $args["email"];
        $password = $args["password"];

        try {
            // TODO: Should these be a part of the context?
            $session = Mage::getSingleton("customer/session");
            // TODO: Attributes?
            $customer = Mage::getModel("customer/customer")
                ->setStore($ctx->getStore())
                ->loadByEmail($email);

            $session->login($email, $password);
            $session->setCustomerAsLoggedIn($customer);

            return new MageQL_Core_Model_Customer_Result_Login(
                MageQL_Core_Model_Customer_Result_Login::SUCCESS,
                $customer
            );
        }
        catch(Mage_Core_Exception $e) {
            if($e->getMessage() === Mage::helper("customer")->__("Invalid login or password.")) {
                return new MageQL_Core_Model_Customer_Result_Login(
                    MageQL_Core_Model_Customer_Result_Login::ERROR_INVALID_LOGIN
                );
            }

            if($e->getMessage() === Mage::helper("customer")->__("This account is not confirmed.")) {
                return new MageQL_Core_Model_Customer_Result_Login(
                    MageQL_Core_Model_Customer_Result_Login::ERROR_NOT_CONFIRMED
                );
            }

            throw $e;
        }
    }

    public static function mutateLogout(): bool {
        $session = Mage::getSingleton("customer/session");

        if($session->isLoggedIn()) {
            $session->logout();

            return true;
        }

        return false;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{customer: array{firstname: string, lastname: string}} $args
     */
    public static function mutateUpdate(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Update {
        $session = Mage::getSingleton("customer/session");
        $form = Mage::getSingleton("mageql/form_customer");

        if( ! $session->isLoggedIn()) {
            return new MageQL_Core_Model_Customer_Result_Update(
                MageQL_Core_Model_Customer_Result_Update::ERROR_NOT_LOGGED_IN
            );
        }

        foreach(array_keys($args["customer"]) as $field) {
            if( ! in_array($field, self::FIELD_SETTINGS)) {
                continue;
            }
            if(
                in_array(
                    $ctx->getConfig(self::FIELD_SETTINGS[$field]),
                    [
                        self::FIELD_READONLY,
                        self::FIELD_HIDDEN
                    ]
                )
            ) {
                unset($args["customer"][$field]);
            }
        }

        $customer = $session->getCustomer();
        $modified = $form->setFields($customer, $args["customer"]);

        if( ! $modified) {
            return new MageQL_Core_Model_Customer_Result_Update(
                MageQL_Core_Model_Customer_Result_Update::NOT_MODIFIED,
                $customer
            );
        }

        $customer->save();

        Mage::dispatchEvent(self::EVENT_MUTATION_UPDATE_CUSTOMER_SUCCESS, [
            "context" => $ctx,
            "customer" => $customer,
        ]);

        return new MageQL_Core_Model_Customer_Result_Update(
            MageQL_Core_Model_Customer_Result_Update::SUCCESS,
            $customer
        );
    }

    /**
     * @param Array<string, string> $fields
     */
        private static function updateCustomer(
        Mage_Customer_Model_Customer $customer,
        array $fields,
        array $data
    ): bool {
        $modified = false;

        foreach($fields as $field => $method) {
            $read = "get$method";
            $write = "set$method";

            if(array_key_exists($field, $data) && $data[$field] !== $customer->$read()) {
                $customer->$write($data[$field]);
                $modified = true;
            }
        }

        return $modified;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{email:string} $args
     */
    public static function mutateUpdateEmail(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_UpdateEmail {
        $session = Mage::getSingleton("customer/session");
        $email = trim($args["email"]);

        if( ! $session->isLoggedIn()) {
            return new MageQL_Core_Model_Customer_Result_UpdateEmail(
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_NOT_LOGGED_IN
            );
        }

        if(
            in_array(
                $ctx->getConfig(self::CONFIG_FIELD_EMAIL),
                [
                    self::FIELD_READONLY,
                    self::FIELD_HIDDEN
                ]
            )
        ) {
            return new MageQL_Core_Model_Customer_Result_UpdateEmail(
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_DISABLED
            );
        }

        $customer = $session->getCustomer();

        if( ! Zend_Validate::is($email, "EmailAddress")) {
            return new MageQL_Core_Model_Customer_Result_UpdateEmail(
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_INVALID_EMAIL_ADDRESS,
                $customer
            );
        }

        if($email === $customer->getEmail()) {
            return new MageQL_Core_Model_Customer_Result_UpdateEmail(
                MageQL_Core_Model_Customer_Result_UpdateEmail::NOT_MODIFIED,
                $customer
            );
        }

        // Save the old email so we can notify customer
        $customer->setOldEmail($customer->getEmail());
        $customer->setEmail($email);
        $customer->setRpToken(null);
        $customer->setRpTokenCreatedAt(null);

        try {
            $customer->save();
        }
        catch(Mage_Customer_Exception $e) {
            if($e->getCode() === Mage_Customer_Model_Customer::EXCEPTION_EMAIL_EXISTS) {
                return new MageQL_Core_Model_Customer_Result_UpdateEmail(
                    MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_EMAIL_EXISTS,
                    $customer
                );
            }

            throw $e;
        }

        $customer->sendChangedPasswordOrEmail();

        // Ensure we update the email in the quote if any
        $checkout = Mage::getSingleton("checkout/session");

        if($checkout->hasQuote()) {
            $quote = $checkout->getQuote();

            $quote->setCustomer($customer);
            $quote->save();
        }

        Mage::dispatchEvent(self::EVENT_MUTATION_UPDATE_CUSTOMER_EMAIL_SUCCESS, [
            "context" => $ctx,
            "customer" => $customer,
        ]);

        return new MageQL_Core_Model_Customer_Result_UpdateEmail(
            MageQL_Core_Model_Customer_Result_UpdateEmail::SUCCESS,
            $customer
        );
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function mutateSetDefaultShippingAddress(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Address_Default {
        return self::doMutateSetDefaultAddress((int)trim($args["id"]), "default_shipping", $ctx);
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function mutateSetDefaultBillingAddress(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Address_Default {
        return self::doMutateSetDefaultAddress((int)trim($args["id"]), "default_billing", $ctx);
    }

    /**
     * @param "default_shipping"|"default_billing" $fieldName
     */
    protected static function doMutateSetDefaultAddress(
        int $addressId,
        string $fieldName,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Address_Default  {
        $session = Mage::getSingleton("customer/session");
        $customer = $session->isLoggedIn() ? $session->getCustomer() : null;

        if( ! $customer) {
            return new MageQL_Core_Model_Customer_Result_Address_Default(
                MageQL_Core_Model_Customer_Result_Address_Default::ERROR_NOT_LOGGED_IN
            );
        }

        if((int)$customer->getData($fieldName) === $addressId) {
            return new MageQL_Core_Model_Customer_Result_Address_Default(
                MageQL_Core_Model_Customer_Result_Address_Default::NOT_MODIFIED
            );
        }

        $address = $customer->getAddressById($addressId);

        if( ! $address->getId()) {
            return new MageQL_Core_Model_Customer_Result_Address_Default(
                MageQL_Core_Model_Customer_Result_Address_Default::ERROR_INVALID_ADDRESS_ID
            );
        }

        $customer->setData($fieldName, $address->getId());
        $customer->save();

        Mage::dispatchEvent(self::EVENT_MUTATION_SET_CUSTOMER_DEFAULT_ADDRESS_SUCCESS, [
            "context" => $ctx,
            "customer" => $customer,
            "address" => $address,
            "type" => $fieldName,
        ]);

        return new MageQL_Core_Model_Customer_Result_Address_Default(
            MageQL_Core_Model_Customer_Result_Address_Default::SUCCESS,
            $customer
        );
    }

    /**
     * Circumventing Magento standard; it does not allow same email on multiple stores,
     * unless we have a Customer enity. This also allow email-oly per store.
     * @param mixed $unusedSrc
     */
    public static function mutateSubsribeToNewsletter(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Newsletter {
        $email = $args["email"] ?? null;
        $firstname = $args["firstname"] ?? null;
        $lastname = $args["lastname"] ?? null;
        $customerSession = Mage::getSingleton("customer/session");
        $isOwnSubscribes = false;
        $storeId = $ctx->getStore()->getId();

        if ($customerSession->isLoggedIn()) {
            // Actual customer
            $customer = $customerSession->getCustomer();
            $email = $email ?? $customer->getEmail();
            $firstname = $firstname ?? $customer->getFirstName();
            $lastname = $lastname ?? $customer->getLastName();
            $isOwnSubscribes = $email == $customer->getEmail();
        } else {
            // Fake customer to be able to find entity by email+store
            if (!$email) {
                return new MageQL_Core_Model_Customer_Result_Newsletter(
                    MageQL_Core_Model_Customer_Result_Newsletter::ERROR_NOT_LOGGED_IN
                );
            }
            $customer = Mage::getModel("customer/customer");
            $customer->setEmail($email);
            $customer->setStoreId($storeId);
        }

        // Validate email
        if (!Zend_Validate::is($email, "EmailAddress")) {
            return new MageQL_Core_Model_Customer_Result_Newsletter(
                MageQL_Core_Model_Customer_Result_Newsletter::ERROR_INVALID_EMAIL
            );
        }

        // Loading by customer id, or email+store if fake customer
        $subscriber = Mage::getModel("newsletter/subscriber")->loadByCustomer($customer);
        if ($subscriber->isSubscribed()) {
            return new MageQL_Core_Model_Customer_Result_Newsletter(
                MageQL_Core_Model_Customer_Result_Newsletter::NOT_MODIFIED
            );
        }

        // If user subscribes own login email - confirmation is not needed
        $isConfirmNeed = (bool)Mage::getStoreConfig(Mage_Newsletter_Model_Subscriber::XML_PATH_CONFIRMATION_FLAG) == 1;
        if ($isConfirmNeed && !$isOwnSubscribes) {
            $subscriber->setStatus(Mage_Newsletter_Model_Subscriber::STATUS_NOT_ACTIVE);
        } else {
            $subscriber->setStatus(Mage_Newsletter_Model_Subscriber::STATUS_SUBSCRIBED);
        }

        if ($isOwnSubscribes) {
            $subscriber->setCustomerId((int)$customer->getId());
        } else {
            $subscriber->setCustomerId(0);
        }

        $subscriber->setStoreId($storeId);
        $subscriber->setSubscriberEmail($email);
        $subscriber->setSubscriberFirstname($firstname);
        $subscriber->setSubscriberLastname($lastname);
        $subscriber->setIsStatusChanged(true);
        $subscriber->save();

        if ($subscriber->isSubscribed()) {
            $subscriber->sendConfirmationSuccessEmail();
        } else {
            $subscriber->sendConfirmationRequestEmail();
        }

        return new MageQL_Core_Model_Customer_Result_Newsletter(
            MageQL_Core_Model_Customer_Result_Newsletter::SUCCESS
        );
    }
}
