<?php

declare(strict_types=1);

namespace MageQL\Core;

use Mage;
use MageQL_Core_Model_Context;
use MageQL_Core_Router_GraphQL;
use ReflectionClass;
use Throwable;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use GraphQL\Utils\SchemaPrinter;
use MageQL\Context;
use PHPUnit\Framework\Assert;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\Driver;
use Spatie\Snapshots\MatchesSnapshots;

/**
 * Special test case which tests the GraphQL schema against a schema.graphql
 * file in the same directory as this file.
 */
class SchemaTest extends TestCase implements Driver {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    protected function getSnapshotId(): string {
        return "schema";
    }

    // Place the snapshot in the integration directoy to make it easy to see
    protected function getSnapshotDirectory(): string {
        return dirname((new ReflectionClass($this))->getFileName());
    }

    // We also implement a test-driver to ensure we use the graphql extension
    public function serialize($data): string {
        /**
         * @var string
         */
        return $data;
    }

    public function extension(): string {
        return "graphql";
    }

    public function match($expected, $actual): void {
        Assert::assertEquals($expected, $this->serialize($actual));
    }

    // Actual test
    public function testSdl(): void {
        MagentoManager::init();

        $app = Mage::app();
        $store = $app->getStore();
        $helper = Mage::helper("mageql/data");
        $layout = Mage::getSingleton("core/layout");

        $layout->setArea(MageQL_Core_Router_GraphQL::GRAPHQL_AREA);
        $app->loadArea($layout->getArea());

        $session = Mage::getSingleton("core/session", ["name" => "frontend"])
            ->start();

        $context = new Context($store, $session, "default");

        $schema = $helper->loadSchema($context, [
            "unreachable" => true,
        ]);

        $sdl = SchemaPrinter::doPrint($schema, [
            "schemaDescriptions" => true,
        ]);

        $this->assertMatchesSnapshot($sdl, $this);
    }
}
