<?php

declare(strict_types=1);

namespace MageQL;

use function array_filter;
use function array_map;
use function get_class;
use function gmdate;
use function lcfirst;
use function method_exists;
use function preg_replace;
use function str_replace;
use function strtolower;
use function strtotime;
use function ucfirst;
use function ucwords;

use Throwable;
use Exception;
use Mage;
use Mage_Core_Model_Abstract;
use Mage_Eav_Model_Entity_Attribute_Source_Abstract;
use Varien_Object;
use Zend_Log;

use GraphQL\Error\Error;
use GraphQL\Type\Definition\ResolveInfo;

/**
 * Forwards the source to the next resolver.
 *
 * @template T
 * @param T $a
 * @return T
 */
function forwardResolver($a) {
    return $a;
}

/**
 * @param string $str
 */
function snakeCaseToCamel($str): string {
    // TODO: Cache?
    return lcfirst(str_replace("_", "", ucwords($str, "_")));
}

/**
 * @param string $str
 */
function spacedToCamel($str): string {
    return str_replace([" ", "-"], ["", "_"], $str);
}

/**
 * @param string $str
 */
function camelToSnakeCase($str): string {
    // TODO: Can we reuse the Varien_Object cache somehow?
    return strtolower(preg_replace("/(.)([A-Z])/", "$1_$2", $str));
}

/**
 * @return ?mixed
 */
function defaultVarienObjectResolver(
    Varien_Object $src,
    array $unusedArgs,
    Context $unusedContext,
    ResolveInfo $info
) {
    $getter = "get".ucfirst($info->fieldName);
    $has = "has".ucfirst($info->fieldName);

    if(method_exists($src, $getter) || $src->$has()) {
        return $src->$getter();
    }

    return null;
}

/**
 * @return mixed
 */
function defaultMethodResolver(
    object $src,
    array $args,
    Context $ctx,
    ResolveInfo $info
) {
    $getter = "get".ucfirst($info->fieldName);

    /**
     * @psalm-suppress MixedMethodCall
     */
    return $src->$getter($args, $ctx, $info);
}

function dateAttributeResolver(
    Varien_Object $src,
    array $args,
    Context $context,
    ResolveInfo $info
): ?string {
    /**
     * @var ?string
     */
    $value = defaultVarienObjectResolver($src, $args, $context, $info);

    if($value !== null) {
        $parsed = strtotime($value);

        return gmdate("Y-m-d\TH:i:s\Z", $parsed);
    }

    return null;
}

function attributeSourceModel(
    Mage_Core_Model_Abstract $src,
    string $name
): ?Mage_Eav_Model_Entity_Attribute_Source_Abstract {
    $attributeCode = camelToSnakeCase($name);

    // Essentially copied from Mage_Catalog_Model_Product::getAttributeText
    // This is missing on Categories and Customer, and here modified to also support
    // multiselect with array-storage
    /** @var \Mage_Eav_Model_Entity_Abstract $resource */
    $resource = $src->getResource();

    $attrResource = $resource->getAttribute($attributeCode);

    // If the attribute no longer exists
    if( ! $attrResource) {
        $error = sprintf(
            "%s: Could not load attribute resource model for '%s' of entity %s in store %d.",
            __FUNCTION__,
            $attributeCode,
            get_class($src),
            Mage::app()->getStore()->getId()
        );

        if(Mage::getIsDeveloperMode()) {
            throw new Exception($error);
        }
        else {
            Mage::log($error, Zend_Log::ERR);
        }

        return null;
    }

    return $attrResource->getSource();
}

function selectAttributeResolver(
    Mage_Core_Model_Abstract $src,
    array $args,
    Context $context,
    ResolveInfo $info
): ?string {
    /**
     * @var ?string
     */
    $value = defaultVarienObjectResolver($src, $args, $context, $info);
    $model = attributeSourceModel($src, $info->fieldName);

    if( ! $model || is_null($value)) {
        return null;
    }

    $text = $model->getOptionText($value);

    return $text !== false ? $text : null;
}

function multiselectAttributeResolver(
    Mage_Core_Model_Abstract $src,
    array $args,
    Context $context,
    ResolveInfo $info
): ?array {
    /**
     * @var Array<string>|string
     */
    $value = defaultVarienObjectResolver($src, $args, $context, $info) ?: "";
    $model = attributeSourceModel($src, $info->fieldName);

    if( ! $model) {
        return null;
    }

    return array_filter(array_map([$model, "getOptionText"], array_map("trim", is_array($value) ? $value : explode(",", $value))));
}

/**
 * Formats errors and appends extensions from ClientException.
 */
function handleGraphQLError(Throwable $error, callable $formatter): array {
    // Default implementation
    /**
     * @var array{extensions:Array<string, mixed>}
     */
    $formatted = $formatter($error);
    $parent = $error->getPrevious();

    if($error instanceof ClientException) {
        $formatted["extensions"] = $error->getExtensions() + $formatted["extensions"] ?? [];
    }

    if($parent instanceof ClientException) {
        $formatted["extensions"] = $parent->getExtensions() + $formatted["extensions"] ?? [];
    }

    return $formatted;
}

/**
 * Sanitizes a graphql name, replacing invalid characters with "_".
 */
function sanitizeGraphqlName(string $name): string {
    $replaced = preg_replace("/[^_0-9A-Za-z]/", "_", $name);

    if(is_numeric($replaced[0] ?? "a")) {
        $replaced[0] = "_";
    }

    return $replaced;
}
