<?php

declare(strict_types=1);

namespace MageQL;

use function array_map;
use function array_merge;
use function sprintf;
use function substr;

use Exception;

use GraphQL\Type\Definition\Type;
use GraphQL\Type\Definition\NonNull;
use GraphQL\Type\Definition\NullableType;
use GraphQL\Type\Definition\ListOfType;
use GraphQL\Type\Schema;

use MageQL\Schema\SchemaInterface;
use MageQL\Type\AbstractBuilder;
use MageQL\Type\FieldBuilder;
use MageQL\Type\InputFieldBuilder;

/**
 * Dynamically expanding type schema.
 */
class Registry {
    /**
     * @var bool
     */
    protected $includeUnreachable;
    /**
     * @var Schema
     */
    protected $schema;
    /**
     * @var Array<SchemaInterface>
     */
    protected $factories = [];

    /**
     * @param Array<SchemaInterface> $factories
     * @param array{includeUnreachable?:bool} $config
     */
    public function __construct(array $factories, array $config) {
        $this->factories = $factories;
        $this->includeUnreachable = $config["includeUnreachable"] ?? false;

        // FIXME: Interfaces break if we dynamically initiate types
        // implementing interfaces.
        // See: https://github.com/webonyx/graphql-php/issues/771
        //
        // $unreachable = $this->includeUnreachable ? $this->createUnreachableTypes() : [];
        $unreachable = $this->createUnreachableTypes();

        $this->schema = new Schema([
            "query" => $this->createType("Query"),
            "mutation" => $this->createType("Mutation"),
            "types" => $unreachable,
            "typeLoader" => function(string $type) {
                return $this->createType($type);
            },
        ]);
    }

    public function getSchema(): Schema {
        return $this->schema;
    }

    /**
     * Retrieves the type specified by the given name, will attempt to create
     * it from the factories in this Registry if it does not exist.
     */
    public function getType(string $typeName): ?Type {
        return $this->schema->getType($typeName);
    }

    /**
     * Creates a type, includes array and strict types.
     */
    protected function createType(string $typeName): Type {
        if(substr($typeName, -1) === "!") {
            $type = $this->schema->getType(substr($typeName, 0, -1));

            if( ! $type) {
                throw new MissingTypeException(substr($typeName, 0, -1));
            }

            return $this->wrapNonNull($type);
        }
        else if(substr($typeName, 0, 1) === "[") {
            if(substr($typeName, -1) !== "]") {
                throw new TypeException(sprintf("Missing closing ']' in type '%s'.", $typeName));
            }

            $type = $this->schema->getType(substr($typeName, 1, -1));

            if( ! $type) {
                throw new MissingTypeException(substr($typeName, 1, -1));
            }

            return $this->wrapListOf($type);
        }
        else {
            return $this->createTypeInstance($typeName);
        }
    }

    /**
     * Returns a map of field-builders for the given type.
     *
     * @return Array<string, FieldBuilder|InputFieldBuilder>
     */
    public function getFieldBuilders(string $typeName): array {
        $fields = [];

        foreach($this->factories as $factory) {
            // TODO: Check if the values are actually FieldBuilders?
            $fields = array_merge($fields, $factory->getTypeFields($typeName, $this));

        }

        return $fields;
    }

    public function getFields(string $typeName): array {
        $fields = $this->getFieldBuilders($typeName);

        return array_map(function($field, string $name): array {
            return $field->createInstance($this, $name);
        }, $fields, array_keys($fields));
    }

    /**
     * @return Array<Type>
     */
    protected function createUnreachableTypes(): array {
        $types = [];
        $typeNames = [];

        foreach($this->factories as $factory) {
            $typeNames = array_merge($typeNames, $factory->getUnreachableTypes());
        }

        foreach($typeNames as $name) {
            $types[] = $this->createTypeInstance($name);
        }

        return $types;
    }

    protected function createTypeInstance(string $typeName): Type {
        foreach($this->factories as $factory) {
            $type = $factory->getTypeBuilder($typeName, $this);

            if($type) {
                return $type->createInstance($this, $typeName);
            }
        }

        throw new MissingTypeException($typeName);
    }

    protected function wrapNonNull(Type $type): Type {
        if( ! $type instanceof NullableType) {
            throw new NotNullableTypeException($type->name);
        }

        $wrapped = new NonNull($type);

        $wrapped->name = $type->name . "!";

        return $wrapped;
    }

    protected function wrapListOf(Type $type): Type {
        $wrapped = new ListOfType($type);

        $wrapped->name = "[" . $type->name . "]";

        return $wrapped;
    }
}
