<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;
use MageQL\Context;

class MageQL_Core_Model_Customer extends Mage_Core_Model_Abstract {
    const NOT_MODIFIED = "notModified";

    /**
     * Event triggered when a customer was successfully updated by the
     * updateCustomer mutation.
     *
     * Parameters:
     *
     *  * context: MageQL\Context
     *  * customer: Mage_Customer_Model_Customer
     */
    const EVENT_MUTATION_UPDATE_CUSTOMER_SUCCESS = "mageql_core_mutation_updateCustomer_success";

    /**
     * Event triggered when a customer email was successfully updated by the
     * updateCustomerEmail mutation.
     *
     * Parameters:
     *
     *  * context: MageQL\Context
     *  * customer: Mage_Customer_Model_Customer
     */
    const EVENT_MUTATION_UPDATE_CUSTOMER_EMAIL_SUCCESS = "mageql_core_mutation_updateCustomerEmail_success";

    /**
     * Event triggered when a customer address default was successfully updated
     * by the setCustomerDefaultAddress mutation.
     *
     * Parameters:
     *
     *  * address: MageQL_Customer_Model_Address
     *  * context: MageQL\Context
     *  * customer: Mage_Customer_Model_Customer
     *  * type: "default_billing"|"default_shipping"
     */
    const EVENT_MUTATION_SET_CUSTOMER_DEFAULT_ADDRESS_SUCCESS = "mageql_core_mutation_setCustomerDefaultAddress_success";

    public static function resolveCustomer(): ?Mage_Customer_Model_Customer {
        $session = Mage::getSingleton("customer/session");

        if($session->isLoggedIn()) {
            return $session->getCustomer();
        }

        return null;
    }

    public static function resolveCreatedAt(
        Mage_Customer_Model_Customer $src
    ): string {
        return gmdate("Y-m-d\TH:i:s\Z", strtotime($src->getCreatedAt()));
    }

    /**
     * @param mixed $unusedSrc
     * @param array{email:string, password:string} $args
     */
    public static function mutateLogin(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Login {
        if( ! $ctx->getConfig(MageQL_Core_Helper_Data::CONFIG_CUSTOMER_STARTUP_ENABLE_LOGIN)) {
            return new MageQL_Core_Model_Customer_Result_Login(
                MageQL_Core_Model_Customer_Result_Login::ERROR_LOGIN_NOT_ENABLED
            );
        }

        $email = $args["email"];
        $password = $args["password"];

        try {
            // TODO: Should these be a part of the context?
            $session = Mage::getSingleton("customer/session");
            // TODO: Attributes?
            $customer = Mage::getModel("customer/customer")
                ->setStore($ctx->getStore())
                ->loadByEmail($email);

            $session->login($email, $password);
            $session->setCustomerAsLoggedIn($customer);

            return new MageQL_Core_Model_Customer_Result_Login(
                MageQL_Core_Model_Customer_Result_Login::SUCCESS,
                $customer
            );
        }
        catch(Mage_Core_Exception $e) {
            if($e->getMessage() === Mage::helper("customer")->__("Invalid login or password.")) {
                return new MageQL_Core_Model_Customer_Result_Login(
                    MageQL_Core_Model_Customer_Result_Login::ERROR_INVALID_LOGIN
                );
            }

            if($e->getMessage() === Mage::helper("customer")->__("This account is not confirmed.")) {
                return new MageQL_Core_Model_Customer_Result_Login(
                    MageQL_Core_Model_Customer_Result_Login::ERROR_NOT_CONFIRMED
                );
            }

            throw $e;
        }
    }

    public static function mutateLogout(): bool {
        $session = Mage::getSingleton("customer/session");

        if($session->isLoggedIn()) {
            $session->logout();

            return true;
        }

        return false;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{customer: array{firstname: string, lastname: string}} $args
     */
    public static function mutateUpdate(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Update {
        $session = Mage::getSingleton("customer/session");
        $form = Mage::getSingleton("mageql/form_customer");

        if( ! $session->isLoggedIn()) {
            return new MageQL_Core_Model_Customer_Result_Update(
                MageQL_Core_Model_Customer_Result_Update::ERROR_NOT_LOGGED_IN
            );
        }

        $customer = $session->getCustomer();
        $modified = $form->setFields($customer, $args["customer"]);

        if( ! $modified) {
            return new MageQL_Core_Model_Customer_Result_Update(
                MageQL_Core_Model_Customer_Result_Update::NOT_MODIFIED,
                $customer
            );
        }

        $customer->save();

        Mage::dispatchEvent(self::EVENT_MUTATION_UPDATE_CUSTOMER_SUCCESS, [
            "context" => $ctx,
            "customer" => $customer,
        ]);

        return new MageQL_Core_Model_Customer_Result_Update(
            MageQL_Core_Model_Customer_Result_Update::SUCCESS,
            $customer
        );
    }

    /**
     * @param Array<string, string> $fields
     */
    private static function updateCustomer(
        Mage_Customer_Model_Customer $customer,
        array $fields,
        array $data
    ): bool {
        $modified = false;

        foreach($fields as $field => $method) {
            $read = "get$method";
            $write = "set$method";

            if(array_key_exists($field, $data) && $data[$field] !== $customer->$read()) {
                $customer->$write($data[$field]);
                $modified = true;
            }
        }

        return $modified;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{email:string} $args
     */
    public static function mutateUpdateEmail(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_UpdateEmail {
        $session = Mage::getSingleton("customer/session");
        $email = trim($args["email"]);

        if( ! $session->isLoggedIn()) {
            return new MageQL_Core_Model_Customer_Result_UpdateEmail(
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_NOT_LOGGED_IN
            );
        }

        $customer = $session->getCustomer();

        if( ! Zend_Validate::is($email, "EmailAddress")) {
            return new MageQL_Core_Model_Customer_Result_UpdateEmail(
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_INVALID_EMAIL_ADDRESS,
                $customer
            );
        }

        if($email === $customer->getEmail()) {
            return new MageQL_Core_Model_Customer_Result_UpdateEmail(
                MageQL_Core_Model_Customer_Result_UpdateEmail::NOT_MODIFIED,
                $customer
            );
        }

        // Save the old email so we can notify customer
        $customer->setOldEmail($customer->getEmail());
        $customer->setEmail($email);
        $customer->setRpToken(null);
        $customer->setRpTokenCreatedAt(null);

        try {
            $customer->save();
        }
        catch(Mage_Customer_Exception $e) {
            if($e->getCode() === Mage_Customer_Model_Customer::EXCEPTION_EMAIL_EXISTS) {
                return new MageQL_Core_Model_Customer_Result_UpdateEmail(
                    MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_EMAIL_EXISTS,
                    $customer
                );
            }

            throw $e;
        }

        $customer->sendChangedPasswordOrEmail();

        // Ensure we update the email in the quote if any
        $checkout = Mage::getSingleton("checkout/session");

        if($checkout->hasQuote()) {
            $quote = $checkout->getQuote();

            $quote->setCustomer($customer);
            $quote->save();
        }

        Mage::dispatchEvent(self::EVENT_MUTATION_UPDATE_CUSTOMER_EMAIL_SUCCESS, [
            "context" => $ctx,
            "customer" => $customer,
        ]);

        return new MageQL_Core_Model_Customer_Result_UpdateEmail(
            MageQL_Core_Model_Customer_Result_UpdateEmail::SUCCESS,
            $customer
        );
    }

    /**
     * @param mixed $unusedSrc
     * @param array{id:string} $args
     */
    public static function mutateSetDefaultShippingAddress(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Address_Default {
        return self::doMutateSetDefaultAddress((int)trim($args["id"]), "default_shipping", $ctx);
    }

    /**
     * @param mixed $unusedSrc
     * @param array{id:string} $args
     */
    public static function mutateSetDefaultBillingAddress(
        $unusedSrc,
        array $args,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Address_Default {
        return self::doMutateSetDefaultAddress((int)trim($args["id"]), "default_billing", $ctx);
    }

    /**
     * @param "default_shipping"|"default_billing" $fieldName
     */
    protected static function doMutateSetDefaultAddress(
        int $addressId,
        string $fieldName,
        Context $ctx
    ): MageQL_Core_Model_Customer_Result_Address_Default  {
        $session = Mage::getSingleton("customer/session");
        $customer = $session->isLoggedIn() ? $session->getCustomer() : null;

        if( ! $customer) {
            return new MageQL_Core_Model_Customer_Result_Address_Default(
                MageQL_Core_Model_Customer_Result_Address_Default::ERROR_NOT_LOGGED_IN
            );
        }

        if((int)$customer->getData($fieldName) === $addressId) {
            return new MageQL_Core_Model_Customer_Result_Address_Default(
                MageQL_Core_Model_Customer_Result_Address_Default::NOT_MODIFIED
            );
        }

        $address = $customer->getAddressById($addressId);

        if( ! $address->getId()) {
            return new MageQL_Core_Model_Customer_Result_Address_Default(
                MageQL_Core_Model_Customer_Result_Address_Default::ERROR_INVALID_ADDRESS_ID
            );
        }

        $customer->setData($fieldName, $address->getId());
        $customer->save();

        Mage::dispatchEvent(self::EVENT_MUTATION_SET_CUSTOMER_DEFAULT_ADDRESS_SUCCESS, [
            "context" => $ctx,
            "customer" => $customer,
            "address" => $address,
            "type" => $fieldName,
        ]);

        return new MageQL_Core_Model_Customer_Result_Address_Default(
            MageQL_Core_Model_Customer_Result_Address_Default::SUCCESS,
            $customer
        );
    }

    /**
     * @param mixed $unusedSrc
     * @param array{email?:?string} $args
     */
    public static function mutateSubsribeToNewsletter(
        $unusedSrc,
        array $args
    ): MageQL_Core_Model_Customer_Result_Newsletter {
        $email = empty($args["email"]) ? null : $args["email"];
        $customerSession = Mage::getSingleton("customer/session");

        if (!$email) {
            if (!$customerSession->isLoggedIn()) {
                return new MageQL_Core_Model_Customer_Result_Newsletter(
                    MageQL_Core_Model_Customer_Result_Newsletter::ERROR_NOT_LOGGED_IN
                );
            }

            $email = Mage::getSingleton("customer/session")->getCustomer()->getEmail();
        }

        if( ! Zend_Validate::is($email, "EmailAddress")) {
            return new MageQL_Core_Model_Customer_Result_Newsletter(
                MageQL_Core_Model_Customer_Result_Newsletter::ERROR_INVALID_EMAIL
            );
        }

        $subscriber = Mage::getModel("newsletter/subscriber")->loadByEmail($email);

        if (!empty($subscriber->getEmail())) {
            return new MageQL_Core_Model_Customer_Result_Newsletter(
                MageQL_Core_Model_Customer_Result_Newsletter::NOT_MODIFIED
            );
        }

        Mage::getModel("newsletter/subscriber")->subscribe($email);

        return new MageQL_Core_Model_Customer_Result_Newsletter(
            MageQL_Core_Model_Customer_Result_Newsletter::SUCCESS
        );
    }
}
