<?php

declare(strict_types=1);

use MageQL\Context;

class MageQL_Core_Model_Address extends Mage_Core_Model_Abstract {
    const MISSING_CITY = "errorMissingCity";
    const MISSING_COUNTRY = "errorMissingCountry";
    const MISSING_FIRSTNAME = "errorMissingFirstname";
    const MISSING_LASTNAME = "errorMissingLastname";
    const MISSING_POSTCODE = "errorMissingPostcode";
    const MISSING_REGION = "errorMissingRegion";
    const MISSING_STREET = "errorMissingStreet";
    const MISSING_TELEPHONE = "errorMissingTelephone";

    public static function resolveCountryCode(Mage_Directory_Model_Country $src): string {
        return $src->getCountryId();
    }

    public static function resolveCountry(
        Mage_Customer_Model_Address_Abstract $src,
        array $unusedArgs,
        Context $ctx
    ): Mage_Directory_Model_Country {
        $model = $src->getCountryModel();

        if( ! $model->getCountryId()) {
            /**
             * This is a mandatory setting.
             *
             * @var string
             */
            $code = $ctx->getStore()->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY);

            $model->loadByCode($code);
        }

        return $model;
    }

    public static function resolveRegion(Mage_Customer_Model_Address_Abstract $src): ?Mage_Directory_Model_Region {
        return $src->getRegionId() ? $src->getRegionModel() : null;
    }

    public static function resolveStreet(Mage_Customer_Model_Address_Abstract $src): array {
        $street = $src->getStreet();
        $parts = is_array($street) ? $street : explode("\n", $street);

        return array_values(array_filter(array_map("trim", $parts)));
    }

    public static function resolveValidationErrors(Mage_Customer_Model_Address_Abstract $address): array {
        /**
         * @var Array<string>|true
         */
        $result = $address->validate();

        if($result === true) {
            return [];
        }

        $translation = self::getErrorTranslation();
        $errors = [];

        foreach($result as $err) {
            if(array_key_exists($err, $translation)) {
                $errors[] = $translation[$err];
            }
            else {
                throw new Exception(sprintf("%s: Unknown address validation error '%s'.", __METHOD__, $err));
            }
        }

        return $errors;
    }

    /**
     * @return Array<string, string>
     */
    public static function getErrorTranslation(): array {
        return [
            Mage::helper("customer")->__("Please enter the city.") => self::MISSING_CITY,
            Mage::helper("customer")->__("Please enter the country.") => self::MISSING_COUNTRY,
            Mage::helper("customer")->__("Please enter the first name.") => self::MISSING_FIRSTNAME,
            Mage::helper("customer")->__("Please enter the last name.") => self::MISSING_LASTNAME,
            Mage::helper("customer")->__("Please enter the state/province.") => self::MISSING_REGION,
            Mage::helper("customer")->__("Please enter the street.") => self::MISSING_STREET,
            Mage::helper("customer")->__("Please enter the telephone number.") => self::MISSING_TELEPHONE,
            Mage::helper("customer")->__("Please enter the zip/postal code.") => self::MISSING_POSTCODE,
        ];
    }
}
