<?php

declare(strict_types=1);

namespace MageQL\Core;

use Mage;
use Varien_Profiler;
use Throwable;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Mage_Core_Model_Store;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class RouteTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function testNotFound(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            route(path: "this/is-not-found") {
                type
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testInternalRedirect(): void {
        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore();

        Mage::getModel("core/url_rewrite")
            ->loadByIdPath("testing-magento-internal")
            ->addData([
                "store_id" => $store->getId(),
                "id_path" => "testing-magento-internal",
                "request_path" => "other-internal-resource",
                "target_path" => "some/internal-resource",
                "is_system" => 0,
                "options" => "R",
            ])
            ->save();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            route(path: "other-internal-resource") {
                type
                ... on RouteRedirect {
                    url
                    isPermanent
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testExteralRedirectUnsecure(): void {
        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore();

        Mage::getModel("core/url_rewrite")
            ->loadByIdPath("testing-magento-external-unsecured")
            ->addData([
                "store_id" => $store->getId(),
                "id_path" => "testing-magento-external-unsecured",
                "request_path" => "external-unsecured",
                "target_path" => "http://some-external.example.com/resource.html",
                "is_system" => 0,
                "options" => "R",
            ])
            ->save();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            route(path: "external-unsecured") {
                type
                ... on RouteRedirect {
                    url
                    isPermanent
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testExteralRedirectSecure(): void {
        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore();

        Mage::getModel("core/url_rewrite")
            ->loadByIdPath("testing-magento-external-secure")
            ->addData([
                "store_id" => $store->getId(),
                "id_path" => "testing-magento-external-secure",
                "request_path" => "external-secure",
                "target_path" => "https://some-external.example.com/resource.html",
                "is_system" => 0,
                "options" => "R",
            ])
            ->save();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            route(path: "external-secure") {
                type
                ... on RouteRedirect {
                    url
                    isPermanent
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
