<?php

declare(strict_types=1);

namespace MageQL\Core;

use Mage;
use Mage_Core_Model_App;
use Mage_Customer_Model_Customer;
use Mage_Core_Model_Store;
use Throwable;
use Varien_Profiler;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Fixtures\Customer;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

require_once __DIR__."/../Fixtures/Customer.php";

class CustomerTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
        Config::unsetConfigPath("default/customer/startup/graphql_enable_login");
        Config::unsetConfigPath("default/customer/changed_account/graphql_email_update_enabled");

        $this->cleanup();
    }

    public function cleanup(): void {
        MagentoManager::init();

        $connection = Mage::getSingleton("core/resource")->getConnection("core_write");

        if($connection) {
            $connection->query("DELETE FROM newsletter_subscriber");
        }

        $fixture = new Customer();

        $fixture->load();
    }

    protected function customerFixture(): Mage_Customer_Model_Customer {
        MagentoManager::init();

        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        return Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");
    }

    public function testCustomerNotLoggedIn(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCustomerLoggedIn(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        vatId
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoginSuccess(): void {
        $this->customerFixture();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                loginCustomer(email: "test-customer@example.com", password: "test-customer") {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoginFailure(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                loginCustomer(email: "test-fail@example.com", password: "test-customer") {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoginDisabled(): void {
        Config::setConfigPath("default/customer/startup/graphql_enable_login", "0");

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                loginCustomer(email: "test-fail@example.com", password: "test-customer") {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLogout(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                logoutCustomer
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLogoutBatch(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            ["query" => 'mutation {
                logoutCustomer
            }'],
            ["query" => 'query {
                customer {
                    firstname
                    email
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateEmailNotLoggedIn(): void {
        $this->customerFixture();

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test@example.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmailNotModified(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test-customer@example.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmailInvalidEmail(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "testexample.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmailExists(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test-duplicate@example.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmail(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test@example.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmailQuote(): void {
        $customer = $this->customerFixture();

        $checkout = Mage::getSingleton("checkout/session");
        $quote = $checkout->getQuote();

        $quote->setCustomerEmail("another@example.com");
        $quote->save();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test@example.com") {
                        result
                    }
                }'));

            $this->assertEquals("test@example.com", $quote->getCustomerEmail());
            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmailDisabled(): void {
        // MageQL_Core_Model_Customer::FIELD_READONLY
        Config::setConfigPath("default/customer/changed_account/graphql_field_email", "readonly");

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test@example.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testSetDefaultShippingAddress(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $address = $customer->getDefaultBillingAddress();

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultShippingAddress(id: '.json_encode($address->getId()).') {
                        result
                    }
                }'));

            $this->assertEquals($customer->getData("default_billing"), $customer->getData("default_shipping"));
            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        vatId
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetDefaultShippingAddressNoop(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $address = $customer->getDefaultShippingAddress();

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultShippingAddress(id: '.json_encode($address->getId()).') {
                        result
                    }
                }'));

            $this->assertEquals($address->getId(), $customer->getData("default_shipping"));
            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testSetDefaultShippingAddressInvalid(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultShippingAddress(id: '.json_encode("0").') {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testSetDefaultShippingAddressNotLoggedIn(): void {
        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultShippingAddress(id: '.json_encode("0").') {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testSetDefaultBillingAddress(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $address = $customer->getDefaultShippingAddress();

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultBillingAddress(id: '.json_encode($address->getId()).') {
                        result
                    }
                }'));

            $this->assertEquals($customer->getData("default_billing"), $customer->getData("default_shipping"));
            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        vatId
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetDefaultBillingAddressNoop(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $address = $customer->getDefaultBillingAddress();

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultBillingAddress(id: '.json_encode($address->getId()).') {
                        result
                    }
                }'));

            $this->assertEquals($address->getId(), $customer->getData("default_billing"));
            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testRemoveCustomerAddress(): void {
        $customer = $this->customerFixture();
        $address = $customer->getDefaultShippingAddress();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                removeCustomerAddress(id: '.json_encode($address->getId()).') {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        vatId
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveCustomerAddressInvalid(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                removeCustomerAddress(id: '.json_encode("0").') {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        vatId
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveCustomerAddressNotLoggedIn(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                removeCustomerAddress(id: '.json_encode("0").') {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateCustomerAddressNotLoggedIn(): void {
        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerAddress(id: '.json_encode("0").', address: {}) {
                        result
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateCustomerAddressNotFound(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerAddress(id: '.json_encode("0").', address: {}) {
                        result
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateCustomerAddressNotModified(): void {
        $customer = $this->customerFixture();
        $billing = $customer->getDefaultBillingAddress();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerAddress(id: '.json_encode($billing->getId()).', address: {
                        firstname: "Test Firstname Billing",
                        lastname: "Test Lastname Billing",
                        city: "Test City Billing",
                        countryCode: "SE",
                        postcode: "54321",
                        street: "Test Street Billing",
                        telephone: "0987654321",
                    }) {
                        result
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateCustomerAddress(): void {
        $customer = $this->customerFixture();
        $billing = $customer->getDefaultBillingAddress();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerAddress(id: '.json_encode($billing->getId()).', address: {
                        firstname: "Test Firstname Billing Modified",
                        lastname: "Test Lastname Billing",
                        city: "Test City Billing",
                        countryCode: "SE",
                        postcode: "12345",
                        street: "Test Street Billing",
                        telephone: "0987654321",
                        vatId: "3517",
                    }) {
                        result
                        address {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            vatId
                            isDefaultBilling
                            isDefaultShipping
                        }
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        vatId
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateCustomerAddressInvalidCountryCode(): void {
        Mage::setIsDeveloperMode(false);

        $customer = $this->customerFixture();
        $billing = $customer->getDefaultBillingAddress();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerAddress(id: '.json_encode($billing->getId()).', address: {
                        firstname: "Test Firstname Billing Modified",
                        lastname: "Test Lastname Billing",
                        city: "Test City Billing",
                        postcode: "12345",
                        street: "Test Street Billing",
                        telephone: "0987654321",
                        countryCode: "invalid country code here",
                    }) {
                        result
                        address {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(400, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCreateCustomerAddress(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    createCustomerAddress(address: {
                        firstname: "Test Added",
                        lastname: "Test The Added ",
                        city: "Test City",
                        countryCode: "SE",
                        postcode: "11111",
                        street: "Test",
                        telephone: "0987654321",
                    }) {
                        result
                        address {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCreateCustomerAddressNotLoggedIn(): void {
        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    createCustomerAddress(address: {
                        firstname: "Test Added",
                        lastname: "Test The Added ",
                        city: "Test City",
                        countryCode: "SE",
                        postcode: "11111",
                        street: "Test",
                        telephone: "0987654321",
                    }) {
                        result
                        address {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testCreateCustomerAddressInvalid(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    createCustomerAddress(address: {
                        firstname: "Test Added",
                        lastname: "Test The Added ",
                        city: "Test City",
                        street: "Test",
                        telephone: "0987654321",
                    }) {
                        result
                        address {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCreateCustomerAddressInvalidCountryCode(): void {
        Mage::setIsDeveloperMode(false);

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    createCustomerAddress(address: {
                        firstname: "Test Added",
                        lastname: "Test The Added ",
                        city: "Test City",
                        postcode: "11111",
                        street: "Test",
                        telephone: "0987654321",
                        countryCode: "some invalid stuff",
                    }) {
                        result
                        address {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(400, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateCustomer(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                updateCustomer(customer: { firstname: "UpdateFirstname", lastname: "UpdateLastname" }) {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        addresses {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                    }
                }
            }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateCustomerNotModified(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                updateCustomer(customer: { firstname: "Test", lastname: "Customer" }) {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        addresses {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                    }
                }
            }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateCustomerNotLoggedIn(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            updateCustomer(customer: { firstname: "Test", lastname: "Customer" }) {
                result
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateCustomerSelect(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                updateCustomer(customer: { gender: "Female" }) {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        gender
                        addresses {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                    }
                }
            }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        // Verify we saved it properly
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            customer {
                email
                prefix
                firstname
                middlename
                lastname
                suffix
                gender
                addresses {
                    firstname
                    lastname
                    street
                    city
                    postcode
                    country {
                        code
                        name
                    }
                    telephone
                    isDefaultBilling
                    isDefaultShipping
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateCustomerSelectBadOption(): void {
        $customer = $this->customerFixture();

        Mage::setIsDeveloperMode(false);

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                updateCustomer(customer: { gender: "Not in list" }) {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        gender
                        addresses {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                    }
                }
            }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(400, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testSubscribeToNewsletterNotLoggedIn(): void {
        MagentoManager::init();

        $email = "foobarbaz@crossroads.se";
        $firstname = "John";
        $lastname = "Doe";
        $subscriber = Mage::getModel("newsletter/subscriber")->loadByEmail($email);

        $subscriber->unsubscribe();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            subscribeToNewsletter(
                email: "'.$email.'",
                firstname: "'.$firstname.'",
                lastname: "'.$lastname.'"
            ) {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSubscribeToNewsletterAlreadyRegisteredOnSameStore(): void {
        MagentoManager::init();

        $email = "foobarbaz@crossroads.se";
        $firstname = "John";
        $lastname = "Doe";

        Mage::getModel("newsletter/subscriber")->subscribe($email);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            subscribeToNewsletter(
                email: "'.$email.'",
                firstname: "'.$firstname.'",
                lastname: "'.$lastname.'"
            ) {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSubscribeToNewsletterAlreadyRegisteredOnOtherStore(): void {
        MagentoManager::init();

        $email = "foobarbaz@crossroads.se";
        $firstname = "John";
        $lastname = "Doe";

        Mage::getModel("newsletter/subscriber")->subscribe($email);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            subscribeToNewsletter(
                email: "'.$email.'",
                firstname: "'.$firstname.'",
                lastname: "'.$lastname.'"
            ) {
                result
            }
        }'), Mage_Core_Model_App::DISTRO_STORE_CODE);

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSubscribeToNewsletterWithoutEmailNotLoggedIn(): void {
        MagentoManager::init();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            subscribeToNewsletter {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSubscribeToNewsletterWithoutEmailLoggedIn(): void {
        MagentoManager::init();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $subscriber = Mage::getModel("newsletter/subscriber")->loadByEmail($customer->getEmail());

        $subscriber->unsubscribe();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            subscribeToNewsletter {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

}
