<?php

declare(strict_types=1);

use GraphQL\Error\Error;
use GraphQL\Language\AST\DocumentNode;
use GraphQL\Language\AST\NodeKind;
use GraphQL\Language\AST\NodeList;
use GraphQL\Language\AST\OperationDefinitionNode;
use GraphQL\Language\AST\SelectionSetNode;
use GraphQL\Language\Visitor;
use GraphQL\Language\VisitorOperation;
use GraphQL\Validator\QueryValidationContext;
use GraphQL\Validator\Rules\QueryComplexity;

/**
 * This class is a copy from GraphQL\Validator\Rules\QueryComplexity with some additions
 */
class MageQL_Core_Model_Validation_QueryComplexity extends QueryComplexity
{
    protected int $warnQueryComplexity;

    /** @throws \InvalidArgumentException */
    public function __construct($maxQueryComplexity = 0, $warnQueryComplexity = 0)
    {
        $this->setMaxQueryComplexity($maxQueryComplexity);
        $this->setWarnQueryComplexity($warnQueryComplexity);
    }

    public function getVisitor(QueryValidationContext $context): array
    {
        $this->queryComplexity = 0;
        $this->context = $context;
        $this->variableDefs = new NodeList([]);
        $this->fieldNodeAndDefs = new \ArrayObject();

        return $this->invokeIfNeeded(
            $context,
            [
                NodeKind::SELECTION_SET => function (SelectionSetNode $selectionSet) use ($context): void {
                    $this->fieldNodeAndDefs = $this->collectFieldASTsAndDefs(
                        $context,
                        $context->getParentType(),
                        $selectionSet,
                        null,
                        $this->fieldNodeAndDefs
                    );
                },
                NodeKind::VARIABLE_DEFINITION => function ($def): VisitorOperation {
                    $this->variableDefs[] = $def;

                    return Visitor::skipNode();
                },
                NodeKind::DOCUMENT => [
                    'leave' => function (DocumentNode $document) use ($context): void {
                        $errors = $context->getErrors();

                        if ($errors !== []) {
                            return;
                        }

                        if (
                            $this->maxQueryComplexity === self::DISABLED &&
                            $this->warnQueryComplexity === self::DISABLED
                        ) {
                            return;
                        }

                        foreach ($document->definitions as $definition) {
                            if (! $definition instanceof OperationDefinitionNode) {
                                continue;
                            }

                            $this->queryComplexity = $this->fieldComplexity($definition->selectionSet);

                            if (
                                $this->warnQueryComplexity !== self::DISABLED &&
                                $this->queryComplexity >= $this->warnQueryComplexity
                            ) {
                                Mage::log(
                                    sprintf('[MageQL] Query complexity: %d', $this->queryComplexity),
                                    Zend_Log::WARN,
                                    MageQL_Core_Helper_Data::LOG_CHANNEL
                                );
                            }

                            if ($this->queryComplexity > $this->maxQueryComplexity) {
                                $context->reportError(
                                    new Error(static::maxQueryComplexityErrorMessage(
                                        $this->maxQueryComplexity,
                                        $this->queryComplexity
                                    ))
                                );

                                return;
                            }
                        }
                    },
                ],
            ]
        );
    }

    public function getWarnQueryComplexity(): int
    {
        return $this->warnQueryComplexity;
    }

    /**
     * Set warn query complexity. Must be greater or equal to 0.
     *
     * @throws \InvalidArgumentException
     */
    public function setWarnQueryComplexity(int $warnQueryComplexity): void
    {
        $this->checkIfGreaterOrEqualToZero('warnQueryComplexity', $warnQueryComplexity);

        $this->warnQueryComplexity = $warnQueryComplexity;
    }

    protected function isEnabled(): bool
    {
        return $this->maxQueryComplexity !== self::DISABLED
            || $this->warnQueryComplexity !== self::DISABLED;
    }
}
