<?php

declare(strict_types=1);

namespace MageQL\Core;

use Mage;
use Exception;
use Throwable;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

require_once __DIR__."/ExceptionTest/Schema.php";

class ExceptionTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
        Config::unsetConfigPath("default/mageql/schema/default/TheTest");
        Config::unsetConfigPath("default/mageql/query/max_complexity");
    }

    public function testMissingModel(): void {
        Config::setConfigPath("default/mageql/schema/default/TheTest/model", "the_missing/schema");

        MagentoManager::init();

        $this->expectException(Exception::class);
        $this->expectExceptionMessage("MageQL_Core_Helper_Data::loadSchema: Model instantiated from config 'config/default/mageql/schema/default/TheTest/model' must be an instance of MageQL\Schema\SchemaInterface");

        MagentoManager::runRequest(new Request("GET /graphql?query={somethingOdd}"));
    }

    public function testMissingModelTag(): void {
        Config::setConfigPath("default/mageql/schema/default/TheTest/somethingElse", "");

        MagentoManager::init();

        $this->expectException(Exception::class);
        $this->expectExceptionMessage("MageQL_Core_Helper_Data::loadSchema: Missing model tag in 'mageql/schema/default/TheTest'.");

        MagentoManager::runRequest(new Request("GET /graphql?query={somethingOdd}"));
    }

    public function testUndefinedMethod(): void {
        Config::setConfigPath("global/models/custom/class", "ExceptionTestModel");
        Config::setConfigPath("default/mageql/schema/default/TheTest/model", "custom/schema");

        MagentoManager::init();

        $res = MagentoManager::runRequest(new Request("GET /graphql?query={somethingOdd}"));

        $log = MagentoManager::readLogFile();

        $this->assertStringContainsString("We failed!", $log);
        $this->assertStringContainsString("Exception", $log);

        $this->assertEquals(500, $res->getHttpResponseCode());
    }

    public function testErrorExtensions(): void {
        Mage::setIsDeveloperMode(false);
        Config::setConfigPath("global/models/custom/class", "ExceptionTestModel");
        Config::setConfigPath("default/mageql/schema/default/TheTest/model", "custom/schema");

        MagentoManager::init();

        $resp = MagentoManager::runRequest(new Request("GET /graphql?query={clientException}"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testGlobalQueryComplexity(): void {
        Mage::setIsDeveloperMode(false);
        Config::setConfigPath("default/mageql/query/max_complexity", "5");

        MagentoManager::init();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                info {
                    baseCurrencyCode
                    baseUrl
                    code
                    currencyCodes
                    locale
                    name
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testFieldQueryComplexity(): void {
        Mage::setIsDeveloperMode(false);
        Config::setConfigPath("default/mageql/schema/default/TheTest/model", "custom/schema");
        Config::setConfigPath("default/mageql/query/max_complexity", "5");

        MagentoManager::init();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                complexField(limit: 6)
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
