<?php

declare(strict_types=1);

namespace MageQL;

use function array_filter;
use function array_map;
use function get_class;
use function gmdate;
use function lcfirst;
use function method_exists;
use function preg_replace;
use function str_replace;
use function strtolower;
use function strtotime;
use function ucfirst;
use function ucwords;

use Throwable;
use Exception;
use Mage;
use Mage_Core_Model_Abstract;
use Mage_Core_Model_Store;
use Mage_Eav_Model_Entity_Attribute_Source_Abstract;
use Varien_Object;
use Zend_Log;

use GraphQL\Error\DebugFlag;
use GraphQL\Error\Error;
use GraphQL\Error\ProvidesExtensions;
use GraphQL\Server\RequestError;
use GraphQL\Error\FormattedError;
use GraphQL\Type\Definition\ResolveInfo;

/**
 * Forwards the source to the next resolver.
 *
 * @template T
 * @param T $a
 * @param array $_args
 * @param mixed $_ctx
 * @param ResolveInfo $_info
 * @return T
 */
function forwardResolver($a, $_args, $_ctx, $_info) {
    return $a;
}

/**
 * @param string $str
 */
function snakeCaseToCamel($str): string {
    // TODO: Cache?
    return lcfirst(str_replace("_", "", ucwords($str, "_")));
}

/**
 * @param string $str
 */
function spacedToCamel($str): string {
    return str_replace([" ", "-"], ["", "_"], $str);
}

/**
 * @param string $str
 */
function camelToSnakeCase($str): string {
    // TODO: Can we reuse the Varien_Object cache somehow?
    return strtolower(preg_replace("/(.)([A-Z])/", "$1_$2", $str));
}

/**
 * @param mixed $unusedContext
 * @return mixed
 */
function defaultVarienObjectResolver(Varien_Object $src, array $unusedArgs, $unusedContext, ResolveInfo $info) {
    $getter = "get".ucfirst($info->fieldName);
    $has = "has".ucfirst($info->fieldName);

    if(method_exists($src, $getter) || $src->$has()) {
        return $src->$getter();
    }

    return null;
}

/**
 * @param mixed $ctx
 * @return mixed
 */
function defaultMethodResolver(object $src, array $args, $ctx, ResolveInfo $info) {
    $getter = "get".ucfirst($info->fieldName);

    return $src->$getter($args, $ctx, $info);
}

/**
 * @param mixed $context
 */
function dateAttributeResolver(Varien_Object $src, array $args, $context, ResolveInfo $info): ?string {
    $value = defaultVarienObjectResolver($src, $args, $context, $info);

    if($value !== null) {
        $parsed = strtotime($value);
        if (!is_int($parsed)) {
            Mage::log("Invalid datetime string: '{$value}'", Zend_Log::WARN);
            return null;
        }

        return gmdate("Y-m-d\TH:i:s\Z", $parsed);
    }

    return null;
}

function attributeSourceModel(Mage_Core_Model_Abstract $src, string $name): ?Mage_Eav_Model_Entity_Attribute_Source_Abstract {
    $attributeCode = camelToSnakeCase($name);

    // Essentially copied from Mage_Catalog_Model_Product::getAttributeText
    // This is missing on Categories and Customer, and here modified to also support
    // multiselect with array-storage
    /** @var \Mage_Eav_Model_Entity_Abstract $resource */
    $resource = $src->getResource();

    $attrResource = $resource->getAttribute($attributeCode);

    // If the attribute no longer exists
    if( ! $attrResource) {
        /** @var Mage_Core_Model_Store $store */
        $store = Mage::app()->getStore();
        $error = sprintf(
            "%s: Could not load attribute resource model for '%s' of entity %s in store %d.",
            __FUNCTION__,
            $attributeCode,
            get_class($src),
            $store->getId()
        );

        if(Mage::getIsDeveloperMode()) {
            throw new Exception($error);
        }
        else {
            Mage::log($error, Zend_Log::ERR);
        }

        return null;
    }

    return $attrResource->getSource();
}

/**
 * @param mixed $context
 */
function selectAttributeResolver(Mage_Core_Model_Abstract $src, array $args, $context, ResolveInfo $info): ?string {
    $value = defaultVarienObjectResolver($src, $args, $context, $info);
    $model = attributeSourceModel($src, $info->fieldName);

    if( ! $model) {
        return null;
    }

    $text = $model->getOptionText($value);

    if($text) {
        return $text;
    }

    return $value;
}

/**
 * @param mixed $args
 * @param mixed $context
 */
function multiselectAttributeResolver(Mage_Core_Model_Abstract $src, $args, $context, ResolveInfo $info): ?array {
    $value = defaultVarienObjectResolver($src, $args, $context, $info);
    $model = attributeSourceModel($src, $info->fieldName);

    if( ! $model || ! $value) {
        return null;
    }

    return array_filter(array_map([$model, "getOptionText"], array_map("trim", is_array($value) ? $value : explode(",", $value))));
}

/**
 * Formats errors and appends extensions from ClientException.
 */
function handleGraphQLError(Throwable $error): array {
    $debug = Mage::getIsDeveloperMode() ?
        DebugFlag::INCLUDE_DEBUG_MESSAGE | DebugFlag::INCLUDE_TRACE :
        DebugFlag::NONE;

    // Default implementation
    /** @var array{extensions?:array} */
    $formatted = FormattedError::createFromException($error, $debug);
    $previous = $error->getPrevious();

    // Request error does not provide extensions
    if($previous instanceof RequestError && empty($formatted["extensions"]["category"])) {
        $formatted["extensions"] = ["category" => "request"] + ($formatted["extensions"] ?? []);
    }

    if($previous instanceof Error && empty($formatted["extensions"]["category"])) {
        $formatted["extensions"] = ["category" => "request"] + ($formatted["extensions"] ?? []);
    }

    if($error instanceof RequestError && empty($formatted["extensions"]["category"])) {
        $formatted["extensions"] = ["category" => "request"] + ($formatted["extensions"] ?? []);
    }

    if($error instanceof Error && empty($formatted["extensions"]["category"])) {
        $formatted["extensions"] = ["category" => "graphql"] + ($formatted["extensions"] ?? []);
    }

    return $formatted;
}

/**
 * Sanitizes a graphql name, replacing invalid characters with "_".
 */
function sanitizeGraphqlName(string $name): string {
    $replaced = preg_replace("/[^_0-9A-Za-z]/", "_", $name);

    if(is_numeric($replaced[0] ?? "a")) {
        $replaced[0] = "_";
    }

    return $replaced;
}
