<?php

declare(strict_types=1);

namespace MageQL\Type;

use function array_key_exists;
use function array_map;

use GraphQL\Type\Definition\ResolveInfo;
use GraphQL\Type\Definition\Type;
use MageQL\Registry;

class FieldBuilder extends TypedBuilder {
    /**
     * @var ?(callable(mixed, array, mixed, ResolveInfo): mixed)
     */
    protected $resolver;
    /**
     * @var Array<string, ArgumentBuilder>
     */
    protected $arguments;
    /**
     * @var ?(callable(mixed, array, mixed, ResolveInfo): int)
     */
    protected $complexity;
    /**
     * @var ?string
     */
    protected $deprecationReason = null;

    /**
     * @param Array<string, ArgumentBuilder> $arguments
     * @param ?(callable(mixed, array, mixed, ResolveInfo):mixed) $resolver
     */
    public function __construct(
        string $typeName,
        string $description,
        array $arguments = [],
        $resolver = null
    ) {
        parent::__construct($typeName, $description);

        $this->arguments = $arguments;
        $this->resolver = $resolver;
    }

    /**
     * @param (callable(mixed, array, mixed, ResolveInfo): mixed) $resolver
     */
    public function setResolver(callable $resolver): self {
        $this->resolver = $resolver;

        return $this;
    }

    /**
     * @return (callable(mixed, array, mixed, ResolveInfo): mixed)
     */
    public function getResolver(): ?callable {
        return $this->resolver;
    }

    /**
     * @param (callable(mixed, array, mixed, ResolveInfo): int) $resolver
     */
    public function setComplexity(callable $complexity): self {
        $this->complexity = $complexity;

        return $this;
    }

    /**
     * @return (callable(mixed, array, mixed, ResolveInfo): int)
     */
    public function getComplexity(): ?callable {
        return $this->complexity;
    }

    /**
     * Deprecates this field with a reason.
     */
    public function setDeprecated(string $reason): self {
        $this->deprecationReason = $reason;

        return $this;
    }

    public function addArgument(string $name, ArgumentBuilder $arg): self {
        $this->arguments[$name] = $arg;

        return $this;
    }

    /**
     * @return array{
     * name:string,
     * description:?string,
     * type:?Type,
     * args:Array<array{name:string, description:?string, type:?Type, defaultValue?:mixed}>,
     * resolve:?callable,
     * complexity:?callable,
     * isDeprecated:bool,
     * deprecationReason:?string,
     * }
     */
    public function createInstance(Registry $registry, string $typeName): array {
        $data = [
            "name" => $typeName,
            "description" => $this->description,
            "type" => $registry->getType($this->type),
            "args" => array_map(function(ArgumentBuilder $arg, string $key) use($registry): array {
                return $arg->createInstance($registry, $key);
            }, $this->arguments, array_keys($this->arguments)),
            "resolve" => $this->resolver,
            "complexity" => $this->complexity,
            "isDeprecated" => (boolean)$this->deprecationReason,
            "deprecationReason" => $this->deprecationReason,
        ];

        return $data;
    }
}
