<?php

declare(strict_types=1);

namespace MageQL;

use function array_map;
use function array_merge;
use function sprintf;
use function substr;

use Exception;

use GraphQL\Type\Definition\Type;
use GraphQL\Type\Definition\NonNull;
use GraphQL\Type\Definition\NullableType;
use GraphQL\Type\Definition\ListOfType;
use GraphQL\Type\Schema;

use MageQL\Schema\SchemaInterface;
use MageQL\Type\AbstractBuilder;
use MageQL\Type\FieldBuilder;
use MageQL\Type\InputFieldBuilder;

/**
 * Dynamically expanding type schema.
 */
class Registry {
    /**
     * @var bool
     */
    protected $includeUnreachable;
    /**
     * @var Schema
     */
    protected $schema;
    /**
     * @var Array<SchemaInterface>
     */
    protected $factories = [];

    /**
     * @param Array<SchemaInterface> $factories
     * @param array{includeUnreachable?:bool} $config
     */
    public function __construct(array $factories, array $config) {
        $this->factories = $factories;
        $this->includeUnreachable = $config["includeUnreachable"] ?? false;

        $unreachable = $this->includeUnreachable ? $this->createUnreachableTypes() : [];

        $this->schema = new Schema([
            "query" => $this->createTypeInstance("Query"),
            "mutation" => $this->createTypeInstance("Mutation"),
            "types" => $unreachable,
            "typeLoader" => function(string $type) {
                return $this->createTypeInstance($type);
            },
        ]);
    }

    public function getSchema(): Schema {
        return $this->schema;
    }

    /**
     * Retrieves the type specified by the given name, will attempt to create
     * it from the factories in this Registry if it does not exist.
     */
    public function getType(string $typeName): ?Type {
        return $this->wrapType($typeName, fn(string $typeName) => $this->schema->getType($typeName));
    }

    /**
     * Wraps the given typeName resolved by typeResolver in the appropriate type-wrappers.
     *
     * @param $typeResolver callable(string $type): ?Type
     */
    protected function wrapType(string $typeName, callable $typeResolver): Type {
        if(substr($typeName, -1) === "!") {
            return new NonNull($this->wrapType(substr($typeName, 0, -1), $typeResolver));
        }
        else if(substr($typeName, 0, 1) === "[") {
            if(substr($typeName, -1) !== "]") {
                throw new TypeException(sprintf("Missing closing ']' in type '%s'.", $typeName));
            }

            return new ListOfType($this->wrapType(substr($typeName, 1, -1), $typeResolver));
        }
        else {
            return $typeResolver($typeName);
        }
    }

    /**
     * Returns a map of field-builders for the given type.
     *
     * @return Array<string, FieldBuilder|InputFieldBuilder>
     */
    public function getFieldBuilders(string $typeName): array {
        $fields = [];

        foreach($this->factories as $factory) {
            // TODO: Check if the values are actually FieldBuilders?
            $fields = array_merge($fields, $factory->getTypeFields($typeName, $this));

        }

        return $fields;
    }

    public function getFields(string $typeName): array {
        $fields = $this->getFieldBuilders($typeName);

        return array_map(function($field, string $name): array {
            return $field->createInstance($this, $name);
        }, $fields, array_keys($fields));
    }

    /**
     * @return Array<Type>
     */
    protected function createUnreachableTypes(): array {
        $types = [];
        $typeNames = [];

        foreach($this->factories as $factory) {
            $typeNames = array_merge($typeNames, $factory->getUnreachableTypes());
        }

        foreach($typeNames as $name) {
            $types[] = $this->createTypeInstance($name);
        }

        return array_filter($types);
    }

    protected function createTypeInstance(string $typeName): ?Type {
        foreach($this->factories as $factory) {
            $type = $factory->getTypeBuilder($typeName, $this);

            if($type) {
                return $type->createInstance($this, $typeName);
            }
        }

        return null;
    }
}
