# Crossroads GraphQL Base Module

Provides routing, configuration and utilties for building GraphQL APIs.

## Registering types and fields

```xml
<config>
  <default>
    <mageql>
      <schema>
        <default> <!-- The name of the schema to add the types to -->
          <ModuleName>
            <model>modulename/schemamodel</model>
          </ModuleName>
        </default>
      </schema>
    </mageql>
  </default>
</config>
```

```php
<?php

declare(strict_types=1);

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

class Modulename_Model_Schemamodel extends MageQL_Model_Schema_Abstract {
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        // Here we define the types we support:
        case "MyType":
            return $this->object("The object type we have added");
        }

        // If $typeName is not a type defined in this schema model:
        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        // Here we define the fields of any object or interface types. We can
        // also define fields for types not defined in this modue which will
        // extend them:
        case "MyType":
            return [
                // Field of type string, uses default getter variant to obtain
                // the value:
                "myfield" => $this->field("String", "A string-field in MyType"),
                "another" => $this->field("Int", "Another field in MyType")
                    ->setResolver(...),
            ];

        case "Query":
            return [
                "mytype" => $this->field("Mytype"),
            ];
        }

        return [];
    }
}
```

### Implementing an interface

```php
public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
    switch($typeName) {
    case "MyInterface":
        return $this->interface("An interface");

    case "MyType":
        return $this->object("The object type we have added")
            ->setInterfaces(["MyInterface"]);
    }

    return null;
}

public function getTypeFields(string $typeName, Registry $registry): array {
    switch($typeName) {
    case "MyInterface":
        // The interface fields, can contain default resolvers and so on:
        return [
            "myfield" => $this->field("String", "A string-field in MyType"),
            "another" => $this->field("Int", "Another field in MyType")
                ->setResolver("resolveAnother"),
        ];

    case "MyType":
        // We use the $registry to fetch the other fields from the
        // interface so we do not have to redefine them:
        return array_merge($registry->getFieldBuilders("MyInterface"), [
            // Fields only existing on this type:
            "mytypefield" => $this->field("String!", "This must exist on MyType"),
        ]);
    }

    return [];
}
```

## Adding attributes to Customer

Available `Customer` fields is controlled by attributes on the `customer`
entity in magento, using the `visible` field.

To add an attribute, use the following in a migration based on
`Mage_Customer_Model_Entity_Setup`:

```php
<?php

declare(strict_types=1);

/**
 * @psalm-scope-this Mage_Customer_Model_Entity_Setup
 */
$setup = $this;

$setup->startSetup();

$entityTypeId = $setup->getEntityTypeId("customer");
$attributeSetId = $setup->getDefaultAttributeSetId($entityTypeId);
$attributeGroupId = $setup->getDefaultAttributeGroupId($entityTypeId, $attributeSetId);

$setup->addAttribute("customer", "attribute_name",  [
    "type" => "varchar",
    "backend"  => "",
    "label" => "Attribute description",
    "input" => "text",
    "source" => "",
    "sort_order" => 250,
    // This will make it show up in GraphQL:
    "visible" => true,
    "required" => false,
    "default" => "",
    "frontend" => "",
    "unique" => false,
    "note" => "Attribute comment",
]);

$setup->addAttributeToGroup(
    $entityTypeId,
    $attributeSetId,
    $attributeGroupId,
    $a,
    300
);

// We need to manually add the attribute here to be able to see it in admin
$attribute = Mage::getSingleton("eav/config")->getAttribute("customer", "attribute_name");

if( ! $attribute) {
    throw new Exception("Could not load customer attribute");
}

$attribute->setData("used_in_forms", ["adminhtml_customer"]);
$attribute->save();

$setup->endSetup();
```

### Adding editable attributes to Customer

To allow the customer to edit an attribute, and add it to the
`UpdateCustomerInput` type, the attribute needs to be in the
`customer_account_edit` form:

```php
<?php

$attribute = Mage::getSingleton("eav/config")->getAttribute("customer", "attribute_name");

if( ! $attribute) {
    throw new Exception("Could not load customer attribute");
}

$attribute->setData("used_in_forms", ["adminhtml_customer", "customer_account_edit"]);
$attribute->save();
```

Note that even if `required` is set to true, the GraphQL schema will not
enforce the presence of the field. This is done later in entity validation.

## Adding attributes to Customer Address

Available `Address` interface and `CustomerAddress` type fields is controlled
by attributes on the `customer_address` entity in magento,
using the `visible` field in the same way fields on `Customer` are managed.

This will also add it to the `QuoteAddress` interface and its implementors,
with `null` as default returned value.

### Adding editable attributes to Customer Address

To allow the customer to edit an attribute, and add it to the
`CustomerAddressInput` type, the attribute needs to be in the
`customer_address_edit` form.

This will also add it to the `QuoteAddressInput` type.

Note that even if `required` is set to true, the GraphQL schema will not
enforce the presence of the field. This is done later in entity validation.

### Adding editable attributes to Quote Address

The quote address inherits attributes from the customer address even if it is
not inheriting the actual database fields. This means that to get a fully
functioning quote address field we have to manually add the database fields
as well as some XML-instructions on how to transform between a customer
address, quote address, and order address.

If these are not added, the fields will be present but always resolve to `null`
and any input data for them will be ignored.

```php
<?php

$conn = $setup->getConnection();

$conn->addColumn($setup->getTable("sales/quote_address"), "attribute_name", [
    "type" => Varien_Db_Ddl_Table::TYPE_TEXT,
    "length" => 255,
    "comment" => "Attribute column description"
));
$conn->addColumn($setup->getTable("sales/order_address"), "attribute_name", [
    "type" => Varien_Db_Ddl_Table::TYPE_TEXT,
    "length" => 255,
    "comment" => "Attribute column description"
));
```

```xml
<config>
  <global>
    <fieldsets>
      <sales_convert_quote_address>
        <attribute_name>
          <to_order_address>*</to_order_address>
        </attribute_name>
      </sales_convert_quote_address>

      <customer_address>
        <attribute_name>
          <to_quote_address>*</to_quote_address>
        </attribute_name>
      </customer_address>
    </fieldsets>
  </global>
</config>
```
