<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

/**
 * @psalm-type AttributeProduct array{0:Mage_Catalog_Model_Resource_Eav_Attribute, 1:Mage_Catalog_Model_Product}
 * @psalm-type ConfigurableItem array{0:Mage_Catalog_Model_Product, 1:Mage_Catalog_Model_Product}
 */
class MageQL_Sales_Model_Item extends Mage_Core_Model_Abstract {
    /**
     * @param Mage_Sales_Model_Quote_Item|Mage_Sales_Model_Order_Item $item
     */
    public static function resolveProduct(Mage_Core_Model_Abstract $item): Mage_Catalog_Model_Product {
        return $item->getProduct();
    }

    /**
     * Resolves the configuration option into a [parent, child] tuple.
     *
     * @return ConfigurableItem
     */
    public static function resolveQuoteConfigOption(Mage_Sales_Model_Quote_Item_Abstract $item): array {
        $children = $item->getChildren();
        $parent = $item->getProduct();
        /**
         * @var Mage_Catalog_Model_Product
         */
        $child = $children[0]->getProduct();

        // Add the parent id so we skip buyRequest on it
        $child->addData([
            "parent_id" => (int)$parent->getId(),
        ]);

        return [$parent, $child];
    }

    /**
     * Resolves the configuration option into a [parent, child] tuple.
     *
     * @return ConfigurableItem
     */
    public static function resolveOrderConfigOption(Mage_Sales_Model_Order_Item $item): array {
        // The only differing line compared to quote:
        $children = $item->getChildrenItems();
        $parent = $item->getProduct();
        /**
         * @var Mage_Catalog_Model_Product
         */
        $child = $children[0]->getProduct();

        // Add the parent id so we skip buyRequest on it
        $child->addData([
            "parent_id" => (int)$parent->getId(),
        ]);

        return [$parent, $child];
    }

    /**
     * @param AttributeProduct $attrProductTuple
     */
    public static function resolveConfigurableAttributeCode($attrProductTuple): string {
        $attr = $attrProductTuple[0];

        // TODO: Should be a bit more reusable
        return snakeCaseToCamel($attr->getAttributeCode());
    }

    /**
     * @param AttributeProduct $attrProductTuple
     */
    public static function resolveConfigurableAttributeLabel($attrProductTuple): string {
        $attr = $attrProductTuple[0];

        return $attr->getStoreLabel() ?: $attr->getFrontend()->getLabel() ?: $attr->getFrontendLabel();
    }

    /**
     * @param AttributeProduct $attrProductTuple
     */
    public static function resolveConfigurableAttributeValue(
        $attrProductTuple,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): string {
        [$attr, $product] = $attrProductTuple;

        // TODO: Can we get this part to actually join the possible attributes when obtaining the collection?
        // We need to load the raw attribute in case it is not set since it
        // might not be joined on the collection because the attribute does
        // not have to be included when querying
        $resource = $product->getResource();
        $model = $attr->getSource();
        // Attempt to load it from the product first, and if it fails then
        // read it from DB
        $value = $product->getData($attr->getAttributeCode());
        /**
         * For second parameter: string in -> string out, array in -> array out
         *
         * @var string|false
         */
        $raw = $resource->getAttributeRawValue((int)$product->getId(), $attr->getAttributeCode(), $ctx->getStore());
        $value = $value ?: $raw;

        if( ! $value) {
            throw new Exception(sprintf(
                "%s: Product %s failed to load attribute %s for config",
                __METHOD__,
                (int)$product->getId(),
                $attr->getAttributeCode()
            ));
        }

        $option = $model->getOptionText($value);

        if( ! is_string($option)) {
            throw new Exception(sprintf(
                "%s: Product %s failed to load attribute %s option text for config option value %s",
                __METHOD__,
                (int)$product->getId(),
                $attr->getAttributeCode(),
                $value
            ));
        }

        return $option;
    }

    /**
     * Uses a [parent, child] tuple as source.
     *
     * @param ConfigurableItem $tuple
     * @return Array<AttributeProduct>
     */
    public static function resolveConfigurableOptionAttributes($tuple): array {
        [$parent, $item] = $tuple;
        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $parent->getTypeInstance(true);
        $attrs = [];

        /**
         * @var Mage_Catalog_Model_Resource_Eav_Attribute $attr
         */
        foreach($instance->getUsedProductAttributes($parent) as $attr) {
            $attrs[] = [$attr, $item];
        }

        return $attrs;
    }

    /**
     * @param ConfigurableItem $tuple
     */
    public static function resolveConfigurableOptionProduct($tuple): Mage_Catalog_Model_Product {
        return $tuple[1];
    }

    public static function resolveOrderBundleOptions(Mage_Sales_Model_Order_Item $item): array {
        $parent = $item->getProduct();
        $children = array_combine(array_map(function($i): int {
            return (int)$i->getProduct()->getId();
        }, $item->getChildrenItems()), array_map(function($i): float {
            return (float)$i->getQtyOrdered();
        }, $item->getChildrenItems()));

        return self::resolveBundleOptions($parent, $children);
    }

    public static function resolveQuoteBundleOptions(Mage_Sales_Model_Quote_Item $item): array {
        $parent = $item->getProduct();
        $children = array_combine(array_map(function(Mage_Sales_Model_Quote_Item $i): int {
            return (int)$i->getProduct()->getId();
        }, $item->getChildren()), array_map(function(Mage_Sales_Model_Quote_Item $i): float {
            return (float)$i->getQty();
        }, $item->getChildren()));

        return self::resolveBundleOptions($parent, $children);
    }

    /**
     * @param Array<int, float> $children product entity id -> quantity
     */
    public static function resolveBundleOptions(Mage_Catalog_Model_Product $parent, array $children): array {
        /**
         * @var Mage_Bundle_Model_Product_Type
         */
        $instance = $parent->getTypeInstance(true);
        $options = $instance->getOptions($parent);

        return array_map(function(
            Mage_Bundle_Model_Option $option
        ) use($children, $parent): MageQL_Sales_Model_Item_Bundle_Option {
            return new MageQL_Sales_Model_Item_Bundle_Option($children, $parent, $option);
        }, array_filter($options, function(Mage_Bundle_Model_Option $o): bool {
            return MageQL_Catalog_Model_Product_Bundle_Option::isVisibleOption($o);
        }));
    }

    /**
     * @param Mage_Sales_Model_Quote_Item|Mage_Sales_Model_Order_Item $src
     */
    public static function resolveRowTotalExVat(
        $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): float {
        $store = $ctx->getStore();

        return (float)$store->roundPrice($src->getRowTotal() - $src->getDiscountAmount());
    }

    /**
     * @param Mage_Sales_Model_Quote_Item|Mage_Sales_Model_Order_Item $src
     */
    public static function resolveRowTotalIncVat(
        $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): float {
        $store = $ctx->getStore();

        return (float)$store->roundPrice($src->getRowTotal()
            + $src->getTaxAmount()
            + $src->getHiddenTaxAmount()
            - $src->getDiscountAmount());
    }

    /**
     * @param Mage_Sales_Model_Quote_Item|Mage_Sales_Model_Order_Item $src
     */
    public static function resolveRowTotalVat($src): float {
        return (float)$src->getTaxAmount();
    }
}
