<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

/**
 * @psalm-suppress PropertyNotSetInConstructor
 */
class MageQL_Sales_Model_Quote_Item extends Mage_Core_Model_Abstract {
    const SUCCESS = "success";
    const ERROR_PRODUCT_NOT_FOUND = "errorProductNotFound";
    const ERROR_PRODUCT_NOT_IN_STOCK = "errorProductNotInStock";
    const ERROR_PRODUCT_QUANTITY_NOT_AVAILABLE = "errorProductQuantityNotAvailable";
    const ERROR_PRODUCT_MAX_QUANTITY = "errorProductMaxQuantity";
    const ERROR_PRODUCT_VARIANT_NOT_AVAILABLE = "errorProductVariantNotAvailable";
    const ERROR_PRODUCT_REQUIRES_BUNDLE_OPTIONS = "errorProductRequiresBundleOptions";
    const ERROR_BUNDLE_SINGLE_GOT_MULTI = "errorBundleSingleGotMulti";
    const ERROR_ITEM_NOT_FOUND = "errorQuoteItemNotFound";
    const ERROR_BUNDLE_EMPTY = "errorBundleEmpty";
    const ERROR_BUNDLE_SELECTION_QTY_IMMUTABLE = "errorBundleSelectionQtyImmutable";
    const ERROR_DECODE = "errorInvalidBuyRequest";

    public static function resolveCanOrder(Mage_Sales_Model_Quote_Item $item): bool {
        /**
         * @var ?Mage_CatalogInventory_Model_Stock_Item|Varien_Object
         */
        $stockItem = $item->getProduct()->getStockItem();
        $qty = $item->getQty();

        if( ! $stockItem) {
            // Assume in stock
            return true;
        }

        /**
         * @var ?Mage_CatalogInventory_Model_Stock_Item|Varien_Object
         */
        $parentStockItem = null;
        /**
         * @var ?Mage_Sales_Model_Quote_Item
         */
        $parentItem = $item->getParentItem();

        if($parentItem) {
            $parentStockItem = $parentItem->getProduct()->getStockItem();
        }

        if( ! $stockItem->getIsInStock() || $parentStockItem && ! $parentStockItem->getIsInStock()) {
            return false;
        }

        $options = $item->getQtyOptions();

        if($options) {
            foreach($options as $option) {
                $optionValue = $option->getValue();
                $optionQty = $qty * $optionValue;

                $optStockItem = $option->getProduct()->getStockItem();

                $optStockItem->setIsChildItem($parentStockItem ? true : false);
                $optStockItem->setSuppressCheckQtyIncrements(true);

                // FIXME: The second optionQty needs to the the cumulative sum
                // of the options of this type in the quote, see
                // Mage_CatalogInventory_Model_Observer::checkQuoteItemQty
                if($optStockItem->checkQuoteItemQty($optionQty, $optionQty, $optionValue)->getHasError()) {
                    return false;
                }

                $optStockItem->unsIsChildItem();
            }
        }
        else {
            $rowQty = $parentItem ? $parentItem->getQty() * $qty : $qty;

            // FIXME: The second rowQty needs to the the cumulative sum
            // of the items of this type in the quote, see
            // Mage_CatalogInventory_Model_Observer::checkQuoteItemQty
            if($stockItem->checkQuoteItemQty($rowQty, $rowQty, $qty)->getHasError()) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{buyRequest:string, qty:float, bundleOptions?: Array<array {
     *   optionId: string, qty?: float, selectionId?: string
     * }>, customOptions?: Array<array {
     *   optionId: string, valueId?: string, field?: string, valueIds?: Array<string>
     * }> } $args
     * @return Mage_Sales_Model_Quote_Item|string
     */
    public static function mutateAdd(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ) {
        $model = Mage::getSingleton("mageql_sales/quote");
        $qty = (float) $args["qty"];

        try {
            $request = MageQL_Sales_Model_BuyRequest::fromString(
                $ctx->getStore(),
                $args["buyRequest"],
                $args["bundleOptions"] ?? [],
                $args["customOptions"] ?? []
            );

            if( ! $request instanceof MageQL_Sales_Model_BuyRequest_Product) {
                return self::ERROR_DECODE;
            }

            $product = $request->getProduct();
            $buyRequest = $request->getBuyRequest();
            // TODO: Deal with custom options (probably additional argument?)
            $buyRequest["qty"] = self::clampProductQty($model->getQuote(), $product, $qty);

            try {
                return $model->addProduct($product, $buyRequest);
            }
            catch(Mage_Core_Exception $e) {
                return self::translateAddProductError($e->getMessage(), $product, $e);
            }
        }
        // TODO: Can we skip most of these? Does frontend even check for all?
        catch(MageQL_Sales_Model_BuyRequest_Exception_ParseError $e) {
            return self::ERROR_DECODE;
        }
        catch(MageQL_Sales_Model_BuyRequest_Exception_BundleSelectionQtyImmutable $e) {
            return self::ERROR_BUNDLE_SELECTION_QTY_IMMUTABLE;
        }
        catch(MageQL_Sales_Model_BuyRequest_Exception_BundleOptionSingleGotMultiple $e) {
            return self::ERROR_BUNDLE_SINGLE_GOT_MULTI;
        }
        catch(MageQL_Sales_Model_BuyRequest_Exception_BundleOptionRequired $e) {
            return self::ERROR_PRODUCT_REQUIRES_BUNDLE_OPTIONS;
        }
        catch(MageQL_Sales_Model_BuyRequest_Exception_BundleMissingOptions $e) {
            return self::ERROR_BUNDLE_EMPTY;
        }
        catch(MageQL_Sales_Model_BuyRequest_Exception_ProductNotFound $e) {
            return self::ERROR_PRODUCT_NOT_FOUND;
        }
    }

    /**
     * @param mixed $unusedSrc
     * @param array{itemBuyRequest:string, qty:float} $args
     * @return Mage_Sales_Model_Quote_Item|string
     */
    public static function mutateUpdateQty(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ) {
        $model = Mage::getSingleton("mageql_sales/quote");
        $qty = (float) $args["qty"];

        try {
            $request = MageQL_Sales_Model_BuyRequest::fromString($ctx->getStore(), $args["itemBuyRequest"]);

            if( ! $request instanceof MageQL_Sales_Model_BuyRequest_Item) {
                return self::ERROR_DECODE;
            }

            $quote = $model->getQuote();
            $item = $request->getItem();
            $product = $item->getProduct();

            // TOOD: Custom options
            // TODO: Move to MageQL_Sales_Model_BuyRequest_Item
            // Join with the existing buy-request to just update the qty
            $buyRequest = array_merge($item->getBuyRequest()->getData(), [
                "product" => $product->getId(),
                "qty" => self::clampProductQty($quote, $product, $qty),
                // Needed to prevent magento from attempting to fetch the original
                // qty + the new qty if we do not have id here the old quote item
                // is considered different and will be first summed with this
                // request and then deleted after the validation has been performed
                // It also needs to be a string to pass === in Mage_Sales_Model_Quote
                // If it is a different item,
                "id" => (string)$item->getId(),
            ]);

            try {
                return $model->updateItem($item, $product, $buyRequest);
            }
            catch(Mage_Core_Exception $e) {
                return self::translateAddProductError($e->getMessage(), $product, $e);
            }
        }
        catch(MageQL_Sales_Model_BuyRequest_Exception_ItemNotFound $e) {
            return self::ERROR_ITEM_NOT_FOUND;
        }
        catch(MageQL_Sales_Model_BuyRequest_Exception_ProductNotFound $e) {
            return self::ERROR_PRODUCT_NOT_FOUND;
        }
    }

    /**
     * @param mixed $unusedSrc
     * @param array{itemBuyRequest:string} $args
     */
    public static function mutateRemove(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): string {
        $model = Mage::getSingleton("mageql_sales/quote");

        try {
            $request = MageQL_Sales_Model_BuyRequest::fromString($ctx->getStore(), $args["itemBuyRequest"]);

            if( ! $request instanceof MageQL_Sales_Model_BuyRequest_Item) {
                return self::ERROR_DECODE;
            }

            $quote = $model->getQuote();
            $item = $request->getItem();

            $quote->removeItem((int)$item->getId());

            // Recollect shipping rates since we have removed an item
            $model->saveSessionQuoteWithShippingRates();

            return self::SUCCESS;
        }
        catch(MageQL_Sales_Model_BuyRequest_Exception_ItemNotFound $e) {
            return self::ERROR_ITEM_NOT_FOUND;
        }
        catch(MageQL_Sales_Model_BuyRequest_Exception_ProductNotFound $e) {
            return self::ERROR_PRODUCT_NOT_FOUND;
        }
    }

    /**
     * Clamps the product quantity if it is not a configurable product.
     *
     * @param float $qty
     */
    public static function clampProductQty(
        Mage_Sales_Model_Quote $quote,
        Mage_Catalog_Model_Product $product,
        $qty
    ): float {
        if($product->isConfigurable()) {
            return $qty;
        }

        /**
         * @var ?Mage_CatalogInventory_Model_Stock_Item
         */
        $stockItem = $product->getStockItem();

        if( ! $stockItem || $quote->hasProductId((int)$product->getId())) {
            return $qty;
        }

        return max($stockItem->getMinSaleQty(), $qty) ?: 0;
    }

    public static function translateAddProductError(
        string $errorMsg,
        Mage_Catalog_Model_Product $product,
        Exception $prev = null
    ): string {
        $error = explode("\n", $errorMsg)[0];

        // TODO: Add option errors if bundle-products need it
        // Option errors cannot happen since we construct the attributes on the server

        if($error === Mage::helper("checkout")->__("The product could not be found.")) {
            return self::ERROR_PRODUCT_NOT_FOUND;
        }

        if($error === Mage::helper('cataloginventory')->__('This product is currently out of stock.')) {
            return self::ERROR_PRODUCT_NOT_IN_STOCK;
        }

        if(stripos($error, Mage::helper('sales')->__('Item qty declaration error.')) !== false ||
           stripos($error, Mage::helper('cataloginventory')->__('Item qty declaration error.\nThe requested quantity for \"%s\" is not available.', $product->getName())) !== false ||
           stripos($error, Mage::helper('cataloginventory')->__('The requested quantity for "%s" is not available.', $product->getName())) !== false) {
           return self::ERROR_PRODUCT_QUANTITY_NOT_AVAILABLE;
        }

        if(stripos($error, strstr(Mage::helper('cataloginventory')->__('The maximum quantity allowed for purchase is %s.', "DEADBEEF"), "DEADBEEF", true) ?: "DEADBEEF") !== false) {
           return self::ERROR_PRODUCT_MAX_QUANTITY;
        }

        if($error === Mage::helper('cataloginventory')->__('This product with current option is not available')) {
            return self::ERROR_PRODUCT_VARIANT_NOT_AVAILABLE;
        }

        throw new Exception($errorMsg, 0, $prev);
    }
}
