<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

/**
 * @psalm-suppress PropertyNotSetInConstructor
 */
class MageQL_Sales_Model_Order extends Mage_Core_Model_Abstract {
    public static function resolveStatus(Mage_Sales_Model_Order $src): string {
        return $src->getStatus();
    }

    public static function resolveEmail(Mage_Sales_Model_Order $src): string {
        return $src->getCustomerEmail();
    }

    public static function resolveLast(): ?Mage_Sales_Model_Order {
        $session = Mage::getSingleton("checkout/session");
        $orderId = $session->getLastOrderId();

        if( ! $orderId) {
            return null;
        }

        $order = Mage::getModel("sales/order");
        $order->load($orderId);

        if( ! $order->getId()) {
            return null;
        }

        return $order;
    }

    public static function resolveR24Amount(Mage_Sales_Model_Order $src): float {
        return (float)$src->getData("retain24_amount");
    }

    /**
     * @return Array<Mage_Sales_Model_Order_Address>
     */
    public static function resolveAddresses(
        Mage_Sales_Model_Order $src
    ): array {
        $addresses = [];

        foreach($src->getAddressesCollection()->getItems() as $addr) {
            if($addr->getAddressType() !== Mage_Customer_Model_Address_Abstract::TYPE_SHIPPING ||
                ! $addr->getSameAsBilling()) {
                $addresses[] = $addr;
            }
        }

        return $addresses;
    }

    public static function resolveCurrencyCode(Mage_Sales_Model_Order $order): string {
        return $order->getOrderCurrencyCode();
    }

    public static function resolveGrandTotal(
        Mage_Sales_Model_Order $order,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): Varien_Object {
        $store = $ctx->getStore();
        $grandTotal = (float)$order->getGrandTotal();
        $tax = (float)$order->getTaxAmount();

        return new Varien_Object([
            "inc_vat" => $grandTotal,
            "ex_vat" => (float) $store->roundPrice($grandTotal - $tax),
            "vat" => $tax,
        ]);
    }

    public static function resolveDiscountTotal(Mage_Sales_Model_Order $order): float {
        return (float)$order->getDiscountAmount();
    }

    public static function resolveId(Mage_Sales_Model_Order $order): string {
        return $order->getIncrementId();
    }

    /**
     * @return Array<Mage_Sales_Model_Order_Item>
     */
    public static function resolveItems(
        Mage_Sales_Model_Order $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $unusedCtx,
        ResolveInfo $info
    ) {
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        /**
         * 2 levels deep to get attributes (1 item, 2 product, 3 attributes)
         *
         * @var array{product:?array{attributes:Array<string, bool>}}
         */
        $fields = $info->getFieldSelection(3);
        // Merge attributes
        $fields = array_merge($fields["product"] ?? [], $fields["product"]["attributes"] ?? []);

        // We use all attribute sets since we cannot be certain about which sets we will get
        $toSelect = $config->getUsedAttributes(
            $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST),
            $fields
        );

        // Set the attributes so they are loaded when we get the quote items
        Mage::getSingleton("mageql_sales/attributes_product")->setQuoteAttributes($toSelect);

        // Do not use a cache since we want to use the previously set attributes
        // Note that the $useCache flag does not work on quote getItemsCollection
        $collection = Mage::getModel("sales/order_item")->getCollection()->setOrderFilter($src);

        $items = $collection->getItems();

        return array_values(array_filter($items, function(Mage_Sales_Model_Order_Item $item) {
            return ! $item->isDeleted() && ! $item->getParentItemId();
        }));
    }

    public static function resolveCreatedAt(Mage_Sales_Model_Order $order): string {
        return gmdate("Y-m-d\TH:i:s\Z", strtotime($order->getCreatedAt()));
    }

    public static function resolveShipping(
        Mage_Sales_Model_Order $src
    ): ?Mage_Sales_Model_Order {
        if( ! $src->getIsVirtual()) {
            return $src;
        }

        return  null;
    }

    public static function resolveSubTotal(
        Mage_Sales_Model_Order $order,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): Varien_Object {
        $store = $ctx->getStore();
        $subTotal = (float)$order->getSubtotalInclTax();
        $subTotalExTax = (float)$order->getSubtotal();

        return new Varien_Object([
            "inc_vat" => $subTotal,
            "ex_vat" => $subTotalExTax,
            "vat" => $store->roundPrice($subTotal - $subTotalExTax),
        ]);
    }

    public static function resolveItemProduct(
        Mage_Sales_Model_Order_Item $item
    ): Mage_Catalog_Model_Product {
        return $item->getProduct();
    }

    public static function resolveVirtual(Mage_Sales_Model_Order $order): bool {
        return (bool) $order->getIsVirtual();
    }
}
