<?php

declare(strict_types=1);

namespace MageQL\Sales\Quote;

use Mage;
use Varien_Object;
use Throwable;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class LowStockTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        $this->reset();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        MagentoManager::logQueries();

        $this->reset();

        throw $t;
    }

    public function reset(): void {
        MagentoManager::reset();
        MagentoManager::init();
        // Reset stock on the product, modified by testPlaceOrderOutOfStock
        $this->setStock([
            "test-simple" => [
                "is_in_stock" => 1,
                "qty" => 999,
            ],
            "test-virtual" => [
                "is_in_stock" => 1,
                "qty" => 999,
            ],
            "test-config-child-1" => [
                "is_in_stock" => 1,
                "qty" => 999,
            ],
            "test-config-child-2" => [
                "is_in_stock" => 1,
                "qty" => 999,
            ],
        ]);

        MagentoManager::reset();
    }

    /**
     * @params Array<string, array{ is_in_stock: bool, qty: int }> $products
     */
    public function setStock(array $products): void {
        foreach($products as $sku => $data) {
            $product = Mage::getModel("catalog/product")
                ->setStoreId(Mage::app()->getStore()->getId())
                ->load(Mage::getModel("catalog/product")->getIdBySku($sku));

            Mage::getModel("cataloginventory/stock_item")
                ->loadByProduct($product)
                ->addData($data)
                ->save();
        }
    }

    public function makeQuote(int $qty = 1): int {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");

        $this->assertNotFalse($attr);

        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        $quote = Mage::getSingleton("checkout/session")->getQuote();
        $productSimple = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $productVirtual = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $productConfig = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($productSimple, new Varien_Object([ "qty" => $qty ]));
        $quote->addProduct($productVirtual, new Varien_Object([ "qty" => $qty ]));
        $quote->addProduct($productConfig, new Varien_Object([
            "qty" => $qty,
            "super_attribute" => [$attr->getId() => $attrBlack],
        ]));
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();

        return (int)$quoteId;
    }

    public function testInStock(): void {
        $quoteId = $this->makeQuote();

        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                items {
                    canOrder
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testNotInStock(): void {
        $quoteId = $this->makeQuote();

        MagentoManager::init();

        $this->setStock([
            "test-simple" => [
                "is_in_stock" => 0,
                "qty" => 999,
            ],
            "test-virtual" => [
                "is_in_stock" => 0,
                "qty" => 999,
            ],
            "test-config-child-2" => [
                "is_in_stock" => 0,
                "qty" => 999,
            ],
        ]);

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                items {
                    canOrder
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLowQuantity(): void {
        $quoteId = $this->makeQuote();

        MagentoManager::init();

        $this->setStock([
            "test-simple" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
            "test-virtual" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
            "test-config-child-2" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
        ]);

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                items {
                    canOrder
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLowQuantity2(): void {
        $quoteId = $this->makeQuote(2);

        MagentoManager::init();

        $this->setStock([
            "test-simple" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
            "test-virtual" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
            "test-config-child-2" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
        ]);

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                items {
                    canOrder
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveAllInStock(): void {
        $quoteId = $this->makeQuote();

        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            removeUnorderableQuoteItems {
                canOrder
                qty
                product {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                items {
                    canOrder
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveNoneInStock(): void {
        $quoteId = $this->makeQuote();

        MagentoManager::init();

        $this->setStock([
            "test-simple" => [
                "is_in_stock" => 0,
                "qty" => 999,
            ],
            "test-virtual" => [
                "is_in_stock" => 0,
                "qty" => 999,
            ],
            "test-config-child-2" => [
                "is_in_stock" => 0,
                "qty" => 999,
            ],
        ]);

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            removeUnorderableQuoteItems {
                canOrder
                qty
                product {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                items {
                    canOrder
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveSimpleNotInStock(): void {
        $quoteId = $this->makeQuote();

        MagentoManager::init();

        $this->setStock([
            "test-simple" => [
                "is_in_stock" => 0,
                "qty" => 999,
            ],
        ]);

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            removeUnorderableQuoteItems {
                canOrder
                qty
                product {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                items {
                    canOrder
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveLowQuantity(): void {
        $quoteId = $this->makeQuote();

        MagentoManager::init();

        $this->setStock([
            "test-simple" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
            "test-virtual" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
            "test-config-child-2" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
        ]);

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            removeUnorderableQuoteItems {
                canOrder
                qty
                product {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                items {
                    canOrder
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveLowQuantity2(): void {
        $quoteId = $this->makeQuote(2);

        MagentoManager::init();

        $this->setStock([
            "test-simple" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
            "test-virtual" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
            "test-config-child-2" => [
                "is_in_stock" => 1,
                "qty" => 1,
            ],
        ]);

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            removeUnorderableQuoteItems {
                canOrder
                qty
                product {
                    sku
                    attributes {
                        shortDescription
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                items {
                    canOrder
                    product {
                        sku
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
