<?php

declare(strict_types=1);

class MageQL_Sales_Helper_Quote {
    /**
     * The checks which the payment method should do when validating.
     */
    const PAYMENT_CHECKS = Mage_Payment_Model_Method_Abstract::CHECK_USE_CHECKOUT
        | Mage_Payment_Model_Method_Abstract::CHECK_USE_FOR_COUNTRY
        | Mage_Payment_Model_Method_Abstract::CHECK_USE_FOR_CURRENCY
        | Mage_Payment_Model_Method_Abstract::CHECK_ORDER_TOTAL_MIN_MAX
        | Mage_Payment_Model_Method_Abstract::CHECK_ZERO_TOTAL;

    /**
     * @throws MageQL_Sales_SubmitOrderException
     */
    public function submitOrder(
        Mage_Sales_Model_Quote $quote
    ): Mage_Sales_Model_Order {
        foreach([$quote->getBillingAddress(), $quote->getShippingAddress()] as $address) {
            $address->setEmail($quote->getCustomerEmail());
        }

        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->save();

        // TODO: Validate agreements
        $this->validateQuote($quote);

        $service = Mage::getModel("sales/service_quote", $quote);

        try {
            $service->submitAll();

            $order = $service->getOrder();
            $redirect = $quote->getPayment()->getOrderPlaceRedirectUrl();

            if( ! $redirect && $order->getCanSendNewEmailFlag()) {
                try {
                    $order->queueNewOrderEmail();
                }
                catch(Exception $e) {
                    Mage::logException($e);
                }
            }

            $quote->setIsActive(0);
            $quote->save();

            return $order;
        }
        catch(Mage_Core_Exception $e) {
            // Convert known Mage_Core_Exception instances which normally happen
            // and are user-visible
            switch($e->getMessage()) {
            case Mage::helper("cataloginventory")
                ->__("Not all products are available in the requested quantity"):
                throw new MageQL_Sales_SubmitOrderException(
                    MageQL_Sales_SubmitOrderException::PRODUCTS_NOT_AVAILABLE
                );
            default:
                throw $e;
            }
        }
        catch(\Exception $e) {
            Mage::logException($e);
            throw new MageQL_Sales_SubmitOrderException(
                MageQL_Sales_SubmitOrderException::UNSPECIFIED
            );
        }
    }

    private function validateQuote(
        Mage_Sales_Model_Quote $quote
    ): void {
        if( ! $quote->hasItems()) {
            throw new MageQL_Sales_SubmitOrderException(
                MageQL_Sales_SubmitOrderException::NO_ITEMS
            );
        }

        $store = $quote->getStore();

        if( ! $quote->isVirtual()) {
            $shipping = $quote->getShippingAddress();
            $validation = $shipping->validate();

            if($validation !== true) {
                // TODO: Propagate shipping address validation errors
                throw new MageQL_Sales_SubmitOrderException(
                    MageQL_Sales_SubmitOrderException::SHIPPING_ADDRESS_VALIDATION_FAILED
                );
            }

            $method = $shipping->getShippingMethod();
            $rate = $shipping->getShippingRateByCode($method);

            if( ! $method || ! $rate) {
                throw new MageQL_Sales_SubmitOrderException(
                    MageQL_Sales_SubmitOrderException::NO_SHIPPING_METHOD
                );
            }

            $approvedIds = Mage::helper("mageql/address")->getAllowedCountryCodes($store);

            if( ! in_array($shipping->getCountryId(), $approvedIds)) {
                throw new MageQL_Sales_InvalidCountryCodeException($shipping->getCountryId());
            }
        }

        $validation = $quote->getBillingAddress()->validate();

        if($validation !== true) {
            // TODO: Propagate billing address validation errors
            throw new MageQL_Sales_SubmitOrderException(
                MageQL_Sales_SubmitOrderException::BILLING_ADDRESS_VALIDATION_FAILED
            );
        }

        $payment = $quote->getPayment();

        if( ! $payment->getMethod()) {
            throw new MageQL_Sales_SubmitOrderException(
                MageQL_Sales_SubmitOrderException::NO_PAYMENT_METHOD
            );
        }

        $paymentSchema = MageQL_Sales_Model_Payment::instance($store);

        if( ! $paymentSchema->hasPaymentMethod($payment->getMethod())) {
            // TODO: Log this as an exception?
            throw new MageQL_Sales_SubmitOrderException(
                MageQL_Sales_SubmitOrderException::INVALID_PAYMENT_METHOD
            );
        }

        // We have to ensure the payment method is still available before placing the order
        $paymentMethod = $payment->getMethodInstance();

        if( ! $paymentMethod->isAvailable($quote) ||
            ! $paymentMethod->isApplicableToQuote(
                $quote,
                self::PAYMENT_CHECKS
            )) {
            throw new MageQL_Sales_SubmitOrderException(
                MageQL_Sales_SubmitOrderException::PAYMENT_NOT_AVAILABLE
            );
        }

        if( ! Zend_Validate::is($quote->getCustomerEmail(), "EmailAddress")) {
            throw new MageQL_Sales_SubmitOrderException(
                MageQL_Sales_SubmitOrderException::INVALID_EMAIL_ADDRESS
            );
        }
    }

    /**
     * Sets the payment method with the supplied data.
     *
     * Example:
     *
     * ```
     * importPaymentData($quote, "free", []);
     * ```
     */
    public function importPaymentData(
        Mage_Sales_Model_Quote $quote,
        string $method,
        array $data
    ): Mage_Sales_Model_Quote_Payment {
        // Shipping totals may be affected by payment method
        if( ! $quote->isVirtual()) {
            $quote->getShippingAddress()
                  ->setCollectShippingRates(true);
        }

        $data["method"] = $method;
        $data["checks"] = self::PAYMENT_CHECKS;

        $payment = $quote->getPayment();

        // Make sure we reset the payment method instance when updating
        if($payment->getMethod() !== $method) {
            $payment->unsetData("method_instance");
        }

        $payment->importData($data);

        return $quote->getPayment();
    }

    /**
     * Attempts to set free if applicable, otherwise it will remove free payment
     * if it is already set but doesn't apply. The new payment instance is
     * returned if free is applied.
     * @psalm-suppress PossiblyUnusedReturnValue
     */
    public function trySetFreeIfAvailable(
        Mage_Sales_Model_Quote $quote
    ): ?Mage_Sales_Model_Quote_Payment {
        $store = $quote->getStore();
        $freeModel = $store->getConfig("payment/free/model");
        $registry = MageQL_Sales_Model_Payment::instance($store);

        if( ! $registry->hasPaymentMethod("free") || ! $freeModel) {
            $this->removeFree($quote);

            return null;
        }

        $free = Mage::getModel($freeModel);

        if( ! $free) {
            $this->removeFree($quote);

            return null;
        }

        $free->setStore($store);

        if( ! $free->isAvailable($quote) || ! $free->isApplicableToQuote($quote, self::PAYMENT_CHECKS)) {
            $this->removeFree($quote);

            return null;
        }

        $payment = $quote->getPayment();

        if($payment->getMethod() === "free") {
            return null;
        }

        return $this->importPaymentData($quote, "free", []);
    }

    private function removeFree(Mage_Sales_Model_Quote $quote): void {
        if($quote->getPayment()->getMethod() === "free") {
            $quote->removePayment();
        }
    }
}
