<?php

declare(strict_types=1);

namespace MageQL\Sales\Quote;

use Mage;
use MageQL_Sales_Model_BuyRequest;
use MageQL_Sales_Model_BuyRequest_Product;
use MageQL_Sales_Model_BuyRequest_Product_Configurable;
use MageQL_Sales_Model_Product;
use MageQL_Sales_Model_Quote_Item;
use Mage_Bundle_Model_Product_Type;
use Mage_Catalog_Model_Product;
use Mage_Catalog_Model_Product_Option;
use Mage_Catalog_Model_Product_Option_Value;
use Mage_Catalog_Model_Product_Type_Configurable;
use RuntimeException;
use Throwable;
use Varien_Object;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class ItemsTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        MagentoManager::logQueries();

        throw $t;
    }

    public function testEmpty(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        product {
                            sku
                            attributes {
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                    }
                    shipping {
                        method {
                            description
                        }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testConfigurable(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");

        $this->assertNotFalse($attr);

        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));

        $request = new Varien_Object([
            "qty" => 1,
            "super_attribute" => [$attr->getId() => $attrBlack],
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal {
                            incVat
                        }
                        product {
                            sku
                            url
                            buyRequest
                            attributes {
                                name
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                        ... on QuoteItemConfigurable {
                            configOption {
                                attributes {
                                    attribute
                                    label
                                    value
                                }
                                product {
                                    sku
                                    buyRequest
                                    attributes {
                                        name
                                        shortDescription
                                        smallImage {
                                            src
                                        }
                                    }
                                }
                            }
                        }
                    }
                    isVirtual
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSimple(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal {
                            incVat
                        }
                        product {
                            sku
                            url
                            attributes {
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                    }
                    isVirtual
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testVirtual(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Virtual requires billing instead of shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal {
                            incVat
                        }
                        product {
                            sku
                            url
                            attributes {
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                    }
                    isVirtual
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddInvalid(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: "{", qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddSimple(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddSimpleChained(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }' ],
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddUnassigned(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-unassigned"));

        // Faked buy request since product is technically not saleable and won't
        // get a buyRequest assigned
        $buyRequest = base64_encode(json_encode([
            "p" => (int)$product->getId(),
        ]));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 1) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $this->assertEquals(false, $quote->hasItems());
    }

    public function testAddVirtual(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddVirtualChained(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }' ],
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddConfigurable(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");

        $this->assertNotFalse($attr);

        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));
        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $product->getTypeInstance(true);
        $attrs = $instance->getConfigurableAttributes($product);
        /**
         * @var ?Mage_Catalog_Model_Product
         */
        $child = $instance->getUsedProductCollection($product)
                ->addAttributeToSelect("*")
                ->addAttributeToFilter("color", $attrBlack)
                ->getFirstItem();

        $this->assertNotNull($child);

        $buyRequest = new MageQL_Sales_Model_BuyRequest_Product_Configurable($product, $child, $attrs);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddConfigVariant(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config-child-1"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 1) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $this->assertEquals(false, $quote->hasItems());
    }

    public function testBuyRequest(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $simpleProduct = Mage::getModel("catalog/product");
        $simpleProduct->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 2,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($simpleProduct, $request);

        // Now add a config
        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");

        $this->assertNotFalse($attr);

        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        // We have to use load, cannot use loadByAttribute or similar
        $configProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));

        $request = new Varien_Object([
            "qty" => 1,
            "super_attribute" => [$attr->getId() => $attrBlack],
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($configProduct, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    items {
                        itemBuyRequest
                        qty
                        product {
                            sku
                        }
                    }
                }
            }
        '));

        $data = json_decode($resp->getBody(), true);

        $this->assertEquals(2, count($data["data"]["quote"]["items"]));
        $first = json_decode(base64_decode($data["data"]["quote"]["items"][0]["itemBuyRequest"]), true);
        $second = json_decode(base64_decode($data["data"]["quote"]["items"][1]["itemBuyRequest"]), true);

        $this->assertArrayHasKey("i", $first);
        $this->assertEquals($first["p"], $simpleProduct->getId());
        $this->assertArrayHasKey("i", $second);
        $this->assertEquals($second["p"], $configProduct->getId());

        // $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleDefault(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-default"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        $childItem = null;

        foreach(Mage::getSingleton("checkout/session")->getQuote()->getAllItems() as $item) {
            if($item->getParentItemId()) {
                $childItem = $item;
            }
        }

        $this->assertNotNull($childItem);

        // Ensure the option title has been loaded for the child item option
        $this->assertStringContainsString("Include extra", $childItem->getOptionsByCode()["bundle_selection_attributes"]->getValue());

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                        ... on QuoteItemBundle {
                            bundleOptions {
                                type
                                title
                                products {
                                    qty
                                    product {
                                        sku
                                        name
                                        attributes {
                                            shortDescription
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleDefaultWithoutOptions(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-default"));

        /**
         * @var Mage_Bundle_Model_Product_Type
         */
        $instance = $product->getTypeInstance(true);
        $storeId = Mage::app()->getStore()->getId();
        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());
        $bundleOptions = array_values(array_filter(array_map(function($o) {
            return [
                "optionId" => $o->getId(),
            ];
        }, $instance->setStoreFilter($storeId, $product)->getOptions($product))));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            "query" => 'mutation ($buyRequest: ID!, $qty: Int!, $bundleOptions: [AddQuoteItemBundleOption!]!) {
                addQuoteItem(buyRequest: $buyRequest, qty: $qty, bundleOptions: $bundleOptions) {
                    result
                }
            }',
            "variables" => [
                "buyRequest" => (string)$buyRequest,
                "qty" => 3,
                "bundleOptions" => $bundleOptions,
            ]
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleSelectWithMultipleOption(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-select"));

        /**
         * @var Mage_Bundle_Model_Product_Type
         */
        $instance = $product->getTypeInstance(true);
        $storeId = Mage::app()->getStore()->getId();
        $buyRequest = base64_encode(json_encode([ "p" => (int)$product->getId() ]));
        $bundleOptions = array_merge(...array_values(array_map(function($o) {
                return array_values(array_map(function($s) use($o) {
                    return [
                        "optionId" => $o->getId(),
                        "qty" => 1,
                        "selectionId" => $s->getSelectionId(),
                    ];
                }, Mage::getResourceModel("bundle/selection_collection")
                    ->setOptionIdsFilter([$o->getId()])
                    ->addStoreFilter(Mage::app()->getStore())
                    ->getItems()
                ));
        }, $instance->setStoreFilter($storeId, $product)->getOptions($product))));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            "query" => 'mutation ($buyRequest: ID!, $qty: Int!, $bundleOptions: [AddQuoteItemBundleOption!]!) {
                addQuoteItem(buyRequest: $buyRequest, qty: $qty, bundleOptions: $bundleOptions) {
                    result
                }
            }',
            "variables" => [
                "buyRequest" => $buyRequest,
                "qty" => 3,
                "bundleOptions" => $bundleOptions,
            ]
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleDefaultWithQty(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-default"));

        /**
         * @var Mage_Bundle_Model_Product_Type
         */
        $instance = $product->getTypeInstance(true);
        $storeId = Mage::app()->getStore()->getId();
        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());
        $bundleOptions = array_values(array_filter(array_map(function($o) {
            return [
                "optionId" => $o->getId(),
                "qty" => 2,
                "selectionId" => array_values(array_map(function($s) {
                        return $s->getSelectionId();
                    }, Mage::getResourceModel("bundle/selection_collection")
                        ->setOptionIdsFilter([$o->getId()])
                        ->addStoreFilter(Mage::app()->getStore())
                        ->getItems()
                    ))[0],
            ];
        }, $instance->setStoreFilter($storeId, $product)->getOptions($product))));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            "query" => 'mutation ($buyRequest: ID!, $qty: Int!, $bundleOptions: [AddQuoteItemBundleOption!]!) {
                addQuoteItem(buyRequest: $buyRequest, qty: $qty, bundleOptions: $bundleOptions) {
                    result
                }
            }',
            "variables" => [
                "buyRequest" => (string)$buyRequest,
                "qty" => 3,
                "bundleOptions" => $bundleOptions,
            ]
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleSelectWithSelectionQty(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-select"));

        /**
         * @var Mage_Bundle_Model_Product_Type
         */
        $instance = $product->getTypeInstance(true);
        $storeId = Mage::app()->getStore()->getId();
        $buyRequest = base64_encode(json_encode([ "p" => (int)$product->getId() ]));
        $bundleOptions = array_values(array_filter(array_map(function($o) {
            return [
                "optionId" => $o->getId(),
                "qty" => 2,
                "selectionId" => array_values(array_map(function($s) {
                        return $s->getSelectionId();
                    }, Mage::getResourceModel("bundle/selection_collection")
                        ->setOptionIdsFilter([$o->getId()])
                        ->addStoreFilter(Mage::app()->getStore())
                        ->getItems()
                    ))[1],
            ];
        }, $instance->setStoreFilter($storeId, $product)->getOptions($product))));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            "query" => 'mutation ($buyRequest: ID!, $qty: Int!, $bundleOptions: [AddQuoteItemBundleOption!]!) {
                addQuoteItem(buyRequest: $buyRequest, qty: $qty, bundleOptions: $bundleOptions) {
                    result
                }
            }',
            "variables" => [
                "buyRequest" => $buyRequest,
                "qty" => 3,
                "bundleOptions" => $bundleOptions,
            ]
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                        ... on QuoteItemBundle {
                            bundleOptions {
                                type
                                title
                                products {
                                    qty
                                    product {
                                        sku
                                        name
                                        attributes {
                                            shortDescription
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleSelectWithNoOptions(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-select"));

        $buyRequest = base64_encode(json_encode([ "p" => (int)$product->getId() ]));
        $bundleOptions = [];

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            "query" => 'mutation ($buyRequest: ID!, $qty: Int!, $bundleOptions: [AddQuoteItemBundleOption!]!) {
                addQuoteItem(buyRequest: $buyRequest, qty: $qty, bundleOptions: $bundleOptions) {
                    result
                }
            }',
            "variables" => [
                "buyRequest" => $buyRequest,
                "qty" => 3,
                "bundleOptions" => $bundleOptions,
            ]
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddCustomOptions(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-custom-options"));

        // standard request
        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddCustomOptionsWithOption(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-custom-options"));


        $select_option = $text_option = null;
        foreach ($product->getProductOptionsCollection() as $option) {
            switch ($option->getType()) {
                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_DROP_DOWN:
                    /** @var Mage_Catalog_Model_Product_Option */
                    $select_option = $option;
                    break;
                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_FIELD:
                    /** @var Mage_Catalog_Model_Product_Option */
                    $text_option = $option;
                    break;
            }
        }
        if (!$select_option || !$text_option) {
            $this->fail("Could not load options for test");
        }

        /** @var Mage_Catalog_Model_Product_Option_Value */
        $value = $select_option->getValuesCollection()->getFirstItem();

        // standard request
        $buyRequest = base64_encode(json_encode([
            "p" => (int)$product->getId(),
        ]));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3, customOptions: [
                    {optionId: ' . (int)$select_option->getId() . ', value: { selectionId: ' . (int)$value->getId() . ' }}
                    {optionId: ' . (int)$text_option->getId() . ', value: { text: "Hello, option" }}
                ]) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                        customOptions {
                            option {
                                title
                                required
                                type
                                sku
                                price {
                                    incVat
                                }
                            }
                            value {
                                selected {
                                    sku
                                    title
                                    price {
                                        incVat
                                    }
                                }
                                text
                            }
                        }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddCustomOptionsRequired(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-custom-options-required"));

        // We make a bad request
        $buyRequest = new MageQL_Sales_Model_BuyRequest_Product($product);

        Mage::setIsDeveloperMode(false);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddCustomOptionsRequiredWithOption(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $value = null;
        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-custom-options-required"));

        $select_option = $text_option = null;
        foreach ($product->getProductOptionsCollection() as $option) {
            switch ($option->getType()) {
                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_DROP_DOWN:
                    $select_option = $option;
                    break;
                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_FIELD:
                    $text_option = $option;
                    break;
            }
        }
        if (!$select_option || !$text_option) {
            $this->fail("Could not load options for test");
        }

        foreach($select_option->getValuesCollection() as $v) {
            if($v->getTitle() === "No") {
                $value = $v;
            }
        }

        if( ! $value) {
            throw new RuntimeException("Failed to load product custom option 'No'.");
        }


        // standard request
        $buyRequest = base64_encode(json_encode([
            "p" => (int)$product->getId(),
        ]));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3, customOptions: [
                    {optionId: ' . (int)$select_option->getId() . ', value: { selectionId: ' . (int)$value->getId() . ' }}
                    {optionId: ' . (int)$text_option->getId() . ', value: { text: "Hello, option" }}
                ]) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                        customOptions {
                            option {
                                title
                                required
                                type
                                sku
                                price {
                                    incVat
                                }
                            }
                            value {
                                selected {
                                    sku
                                    title
                                    price {
                                        incVat
                                    }
                                }
                                text
                            }
                        }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddCustomOptionsRequiredWithOptionYes(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $value = null;
        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-custom-options-required"));

        $select_option = $text_option = null;
        foreach ($product->getProductOptionsCollection() as $option) {
            switch ($option->getType()) {
                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_DROP_DOWN:
                    $select_option = $option;
                    break;
                case Mage_Catalog_Model_Product_Option::OPTION_TYPE_FIELD:
                    $text_option = $option;
                    break;
            }
        }
        if (!$select_option || !$text_option) {
            $this->fail("Could not load options for test");
        }

        foreach($select_option->getValuesCollection() as $v) {
            if($v->getTitle() === "Yes") {
                $value = $v;
            }
        }

        if( ! $value) {
            throw new RuntimeException("Failed to load product custom option 'Yes'.");
        }

        $buyRequest = base64_encode(json_encode([
            "p" => (int)$product->getId(),
        ]));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3, customOptions: [
                    {optionId: ' . (int)$select_option->getId() . ', value: { selectionId: ' . (int)$value->getId() . ' }}
                    {optionId: ' . (int)$text_option->getId() . ', value: { text: "Hello, option" }}
                ]) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                        customOptions {
                            option {
                                title
                                required
                                type
                                sku
                                price {
                                    incVat
                                }
                            }
                            value {
                                selected {
                                    sku
                                    title
                                    price {
                                        incVat
                                    }
                                }
                                text
                            }
                        }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateQty(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        // We have to use load, cannot use loadByAttribute or similar
        $simpleProduct = Mage::getModel("catalog/product");
        $simpleProduct->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 2,
        ]);
        $simpleItem = $quote->addProduct($simpleProduct, $request);

        $this->assertIsObject($simpleItem);

        // Now add a config
        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");

        $this->assertNotFalse($attr);

        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        // We have to use load, cannot use loadByAttribute or similar
        $configProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));

        $request = new Varien_Object([
            "qty" => 1,
            "super_attribute" => [$attr->getId() => $attrBlack],
        ]);

        $configItem = $quote->addProduct($configProduct, $request);

        $this->assertIsObject($configItem);

        $parentItem = $configItem->getParentItem();

        $storeId = Mage::app()->getStore()->getId();
        $bundleProduct = Mage::getModel("catalog/product");
        $bundleProduct->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-default"));

        /**
         * @var Mage_Bundle_Model_Product_Type
         */
        $instance = $bundleProduct->getTypeInstance(true);
        $bundleOptions = $instance->setStoreFilter($storeId, $bundleProduct)
            ->getOptions($bundleProduct);

        $request = new Varien_Object([
            "qty" => 1,
            "bundle_option" => array_combine(
                array_map(function($o) { return $o->getId(); }, $bundleOptions),
                array_map(function($o) {
                    return array_values(array_map(function($s) {
                        return $s->getSelectionId();
                    }, Mage::getResourceModel("bundle/selection_collection")
                        ->setOptionIdsFilter([$o->getId()])
                        ->addStoreFilter(Mage::app()->getStore())
                        ->getItems()
                    ));
                }, $bundleOptions)
            ),
            "bundle_option_qty" => array_combine(
                array_map(function($o) { return $o->getId(); }, $bundleOptions),
                array_map(function($unusedO) { return 1; }, $bundleOptions)
            ),
        ]);

        $item = $quote->addProduct($bundleProduct, $request);

        $this->assertIsObject($item);

        $bundleItem = $item->getParentItem();

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        assert($parentItem !== null);
        assert($bundleItem !== null);

        $buyRequestSimple = MageQL_Sales_Model_BuyRequest::fromQuoteItem($simpleItem);
        $buyRequestConfig = MageQL_Sales_Model_BuyRequest::fromQuoteItem($parentItem);
        $buyRequestBundle = MageQL_Sales_Model_BuyRequest::fromQuoteItem($bundleItem);

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                updateQuoteItemQty(itemBuyRequest: '.json_encode($buyRequestSimple).', qty: 4) {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                updateQuoteItemQty(itemBuyRequest: '.json_encode($buyRequestConfig).', qty: 2) {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                updateQuoteItemQty(itemBuyRequest: '.json_encode($buyRequestBundle).', qty: 3) {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateQtyHiddenBundle(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $storeId = Mage::app()->getStore()->getId();
        $bundleProduct = Mage::getModel("catalog/product");
        $bundleProduct->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-hidden"));

        /**
         * @var Mage_Bundle_Model_Product_Type
         */
        $instance = $bundleProduct->getTypeInstance(true);
            // Prepare store filter for correct option title data
        $bundleOptions = $instance->setStoreFilter($storeId, $bundleProduct)
            ->getOptions($bundleProduct);

        $request = new Varien_Object([
            "qty" => 1,
            "bundle_option" => array_combine(
                array_map(function($o) { return $o->getId(); }, $bundleOptions),
                array_map(function($o) {
                    return array_values(array_map(function($s) {
                        return $s->getSelectionId();
                    }, Mage::getResourceModel("bundle/selection_collection")
                        ->setOptionIdsFilter([$o->getId()])
                        ->addStoreFilter(Mage::app()->getStore())
                        ->getItems()
                    ));
                }, $bundleOptions)
            ),
            "bundle_option_qty" => array_combine(
                array_map(function($o) { return $o->getId(); }, $bundleOptions),
                array_map(function($unusedO) { return 2; }, $bundleOptions)
            ),
        ]);

        $item = $quote->addProduct($bundleProduct, $request);

        $this->assertIsObject($item);

        $bundleItem = $item->getParentItem();

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        assert($bundleItem !== null);

        $buyRequestBundle = MageQL_Sales_Model_BuyRequest::fromQuoteItem($bundleItem);

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                updateQuoteItemQty(itemBuyRequest: '.json_encode($buyRequestBundle).', qty: 3) {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
        ])));

        $quote = Mage::getModel("sales/quote")->load($quoteId);

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        // Ensure we did not duplicate quote items, should be one parent and one child:
        $this->assertEquals(2, count($quote->getAllItems()));
    }

    public function testRemove(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $simpleProduct = Mage::getModel("catalog/product");
        $simpleProduct->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 2,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $simpleItem = $quote->addProduct($simpleProduct, $request);

        $this->assertIsObject($simpleItem);

        // Now add a config
        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");

        $this->assertNotFalse($attr);

        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        // We have to use load, cannot use loadByAttribute or similar
        $configProduct = Mage::getModel("catalog/product");
        $configProduct->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));

        $request = new Varien_Object([
            "qty" => 1,
            "super_attribute" => [$attr->getId() => $attrBlack],
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $item = $quote->addProduct($configProduct, $request);

        $this->assertIsObject($item);

        $configItem = $item->getParentItem();

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        assert($configItem !== null);

        $buyRequestSimple = MageQL_Sales_Model_BuyRequest::fromQuoteItem($simpleItem);
        $buyRequestConfig = MageQL_Sales_Model_BuyRequest::fromQuoteItem($configItem);

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                removeQuoteItem(itemBuyRequest: '.json_encode($buyRequestSimple).') {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                removeQuoteItem(itemBuyRequest: '.json_encode($buyRequestConfig).') {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateQtyAndId(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $simpleProduct = Mage::getModel("catalog/product");
        $simpleProduct->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 2,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $simpleItem = $quote->addProduct($simpleProduct, $request);

        $this->assertIsObject($simpleItem);

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        $unassigned = Mage::getModel("catalog/product");
        $unassigned->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-unassigned"));

        // Faked buy request since product is technically not saleable and won't
        // get a buyRequest assigned, and we construct an invalid buy-request
        // in the attempt to replace the product
        $buyRequestSimple = base64_encode(json_encode([
            "i" => (int)$simpleItem->getId(),
            "p" => (int)$unassigned->getId(),
        ]));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                updateQuoteItemQty(itemBuyRequest: '.json_encode($buyRequestSimple).', qty: 4) {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
