<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Sales_Model_Payment {
    /**
     * @var ?Array<MageQL_Sales_Model_Payment_Abstract>
     */
    protected $models = null;

    /**
     * @var Mage_Core_Model_Store
     */
    protected $store;

    public static function instance(Mage_Core_Model_Store $store): self {
        $id = sprintf("mageql_sales_payment_%d", $store->getId());

        $instance = Mage::registry($id);

        if(!$instance) {
            $instance = new self($store);

            Mage::register($id, $instance);
        }

        return $instance;
    }

    private function __construct(Mage_Core_Model_Store $store) {
        $this->store = $store;
    }

    /**
     * Returns a list of enabled payment methods for the current store.
     *
     * @return Array<MageQL_Sales_Model_Payment_Abstract>
     */
    public function getModels(): array {
        if($this->models === null) {
            $this->models = [];

            /**
             * @var Array<string, array{model:string}>
             */
            $payments = $this->store->getConfig("mageql/sales/payment");

            foreach($payments as $code => $data) {
                /**
                 * @var MageQL_Sales_Model_Payment_Abstract|null
                 */
                $model = Mage::getSingleton($data["model"]);

                if( ! $model instanceof MageQL_Sales_Model_Payment_Abstract) {
                    throw new Exception(sprintf(
                        "%s: MageQL sales payment model '%s' implementing %s for method with code '%s' could not be found",
                        __METHOD__,
                        $data["model"],
                        MageQL_Sales_Model_Payment_Abstract::class,
                        $code
                    ));
                }

                $code = $model->getCode();

                // We do not want to include inactives since that will give the
                // wrong impression to clients if types for unsupported payment
                // types exist
                if($this->store->getConfig("payment/$code/active")) {
                    $this->models[$code] = $model;
                }
            }
        }

        return $this->models;
    }

    /**
     * Returns true if this schema has support for the supplied payment method.
     */
    public function hasPaymentMethod(string $code): bool {
        foreach($this->getModels() as $m) {
            if($m->getCode() === $code) {
                return true;
            }
        }

        return false;
    }

    public function typePaymentMethod(Mage_Payment_Model_Method_Abstract $src): string {
        $match = $this->getModels()[$src->getCode()] ?? null;

        if( ! $match) {
            // This should not happen unless paymentMethods resolver is broken
            throw new Exception(sprintf("%s: Unsupported payment method '%s'", __METHOD__, $src->getCode()));
        }

        return $match->getPaymentMethodType();
    }

    public function typeQuotePayment(Mage_Sales_Model_Quote_Payment $src): string {
        $match = $this->getModels()[$src->getMethod()] ?? null;

        if( ! $match) {
            // This should not happen unless it is a broken quote, or the quote payment resolver is broken
            throw new Exception(sprintf("%s: Unsupported payment method '%s'", __METHOD__, $src->getMethod()));
        }

        return $match->getQuotePaymentType();
    }

    /**
     * @return Array<string, array{errorCode?:int, description:string}>
     */
    public function getPlaceOrderErrors(): array {
        $models = $this->getModels();
        $errors = [];

        foreach($models as $m) {
            $errors = array_merge($errors, $m->getPlaceOrderErrors());
        }

        return $errors;
    }
}
