<?php

declare(strict_types=1);

namespace MageQL\Sales\Quote;

use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Customer;
use MageQL_Sales_Model_BuyRequest;
use Mage_Core_Helper_Data;
use Throwable;
use Varien_Object;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class InvalidCountryTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
        MagentoManager::initAdmin();

        // Testing tax rates do not match NL, and is not in accepted countries in testing store
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        foreach($customer->getAddresses() as $address) {
            $address->setCountryId("NL");
            $address->save();
        }

        $setupModel->setConfigData("carriers/freeshipping/active", 1, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::reset();
        MagentoManager::initAdmin();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        foreach($customer->getAddresses() as $address) {
            $address->setCountryId("SE");
            $address->save();
        }

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->deleteConfigData("carriers/freeshipping/active", "websites");
        $setupModel->deleteConfigData("carriers/freeshipping/free_shipping_subtotal", "websites");

        // Reset stock on the product, modified by testPlaceOrderOutOfStock
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        Mage::getModel("cataloginventory/stock_item")
            ->loadByProduct($product)
            ->addData([
                "is_in_stock" => 1,
                "qty" => 999,
            ])
            ->save();

        // we have to reindex the stock to not affect the other tests
        MagentoManager::reset();
        MagentoManager::reindex();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        $this->tearDown();

        throw $e;
    }

    protected function customerFixture(): Mage_Customer_Model_Customer {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        return Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");
    }

    public function testDefaultCountry(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // Prevent any existing quotes from being loaded
        $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

        if($conn) {
            $conn->query("UPDATE sales_flat_quote SET is_active = 0");
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $session = Mage::getSingleton("customer/session");

        $session->login("test-customer@example.com", "test-customer");
        $customer->setSession($session->getSessionId());

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal { incVat exVat vat }
                    discountTotal
                    grandTotal { incVat exVat vat }
                    items {
                        qty
                        rowTotal { incVat exVat vat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetCountry(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // Prevent any existing quotes from being loaded
        $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

        if($conn) {
            $conn->query("UPDATE sales_flat_quote SET is_active = 0");
        }

        MagentoManager::reset();
        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        Mage::setIsDeveloperMode(false);
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
                setQuoteShippingAddress(address: {
                    countryCode: "NL"
                }) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal { incVat exVat vat }
                    discountTotal
                    grandTotal { incVat exVat vat }
                    items {
                        qty
                        rowTotal { incVat exVat vat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderFail(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "NL",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "NL",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        Mage::setIsDeveloperMode(false);
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrder(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $item = $quote->addProduct($product, $request);

        $this->assertIsObject($item);

        $item->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
                ...on PlaceOrderResultSuccess {
                    paymentRedirectUrl
                    order {
                        status
                        items {
                            qty
                            rowTotal { incVat exVat vat }
                            product {
                                sku
                                attributes {
                                    shortDescription
                                    smallImage {
                                        src
                                    }
                                }
                            }
                        }
                        addresses {
                            type
                            prefix
                            firstname
                            middlename
                            lastname
                            suffix
                            company
                            street
                            city
                            postcode
                            region {
                                code
                                name
                            }
                            country {
                                code
                            }
                        }
                        email
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
