<?php

declare(strict_types=1);

namespace MageQL\Sales;

use Mage;
use MageQL_Core_Model_Context;
use MageQL_Sales_Model_BuyRequest;
use MageQL_Sales_Model_Product;
use Mage_Catalog_Model_Product_Status;
use Throwable;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class ProductTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testBuyRequestUnassigned(): void {
        MagentoManager::init();

        $product = Mage::getModel("catalog/product");

        if ($entityId = $product->getIdBySku("test-unassigned")) {
            $product->load($entityId);
        }

        $this->assertNull(MageQL_Sales_Model_Product::resolveBuyRequestSimple($product, [], new MageQL_Core_Model_Context([
            "store" => Mage::app()->getStore(),
            "session" => null,
        ])));
    }

    public function testBuyRequestConfigurableDetail(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-config") {
                sku
                type
                buyRequest
                ... on ProductDetailConfigurable {
                    configOptions {
                        attributes { attribute label }
                        items {
                            buyRequest
                            product {
                                sku
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBuyRequestSimpleDetail(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-simple") {
                sku
                type
                buyRequest
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBuyRequestVirtualDetail(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-virtual") {
                sku
                type
                buyRequest
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBuyRequestBundleDefault(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-bundle-default") {
                sku
                type
                buyRequest
                ... on ProductDetailBundle {
                    bundleBuyRequestBase
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBuyRequestList(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            categoryNav {
                name
                products {
                    totalCount
                    items {
                        sku
                        name
                        buyRequest
                        ... on ListProductConfigurable {
                            configOptions {
                                attributes { attribute label }
                                items {
                                    buyRequest
                                    product {
                                        sku
                                    }
                                }
                            }
                        }
                        ... on ListProductBundle {
                            bundleBuyRequestBase
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBuyRequestBundleSelect(): void {
        MagentoManager::init();

        $product = Mage::getModel("catalog/product");

        if ($entityId = $product->getIdBySku("test-bundle-select")) {
            $product->load($entityId);
        }

        $this->assertNull(MageQL_Sales_Model_Product::resolveBuyRequestSimple($product, [], new MageQL_Core_Model_Context([
            "store" => Mage::app()->getStore(),
            "session" => null,
        ])));
    }

    public function testBuyRequestBundleDisabled(): void {
        MagentoManager::init();

        $product = Mage::getModel("catalog/product");

        if ($entityId = $product->getIdBySku("test-bundle-default")) {
            $product->load($entityId);
        }

        $product->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);

        $this->assertNull(MageQL_Sales_Model_Product::resolveBuyRequestSimple($product, [], new MageQL_Core_Model_Context([
            "store" => Mage::app()->getStore(),
            "session" => null,
        ])));
    }

    public function testBuyRequestCustomOptions(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-custom-options") {
                sku
                customOptions {
                    title
                    values {
                        sku
                        title
                        buyRequest
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBuyRequestCustomOptionsRequired(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku (sku: "test-custom-options-required") {
                sku
                customOptions {
                    title
                    values {
                        sku
                        title
                        buyRequest
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testBuyRequestCustomOptionsRequired2(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-custom-options-required"));

        // standard request
        $this->assertNull(MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore()));
    }
}
