<?php

declare(strict_types=1);

namespace MageQL\Sales\Discount;

use Mage;
use Throwable;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

use MageQL_Sales_Model_BuyRequest_Product;

class DiscountTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        parent::setUp();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    static function setUpBeforeClass(): void {
        MagentoManager::reset();
        MagentoManager::init();

        $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

        if($conn) {
           $conn->query("DELETE FROM sales_flat_quote")->execute();
        }
    }

    static function resetAndInitButKeepQuote(): void {
        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();
        MagentoManager::reset();
        MagentoManager::init();
        if (!empty($quoteId)) {
            Mage::getSingleton("checkout/session")->setQuoteId($quoteId);
        }
    }

    public function testAddDiscountCodeFail(): void {

        /*
         *  Step 1: Add product "test-simple" to quote
         */
        MagentoManager::reset();
        MagentoManager::init();

        $product1 = Mage::getModel("catalog/product");
        $product1->load($product1->getIdBySku("test-simple"));
        $buyRequest1 = json_encode(new MageQL_Sales_Model_BuyRequest_Product($product1));

        $res1 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            addQuoteItem(buyRequest:' . $buyRequest1 . ',qty:1) { result }
        }'));

        $this->assertMatchesJsonSnapshot($res1->getBody());
        $this->assertEquals(200, $res1->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res1->getHeader("Content-Type"));

        /*
         *  Step 2: Apply discount code "total-failure"
         *  Make sure it fails
         */
        self::resetAndInitButKeepQuote();
        $res2 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            setQuoteDiscountCode(code:"total-failure") { result, coupon { code, label } }
        }'));

        $this->assertMatchesJsonSnapshot($res2->getBody());
        $this->assertEquals(200, $res2->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res2->getHeader("Content-Type"));

        /*
         *  Step 3: Fetch quote info
         *  Make sure no discount is applied
         */
        self::resetAndInitButKeepQuote();
        $res3 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            quote {
                items { product { sku, price { incVat } } }
                coupon { code, label }
                discounts { label, type }
                grandTotal { incVat }
                subTotal { incVat }
                discountTotal
            }
        }'));

        $this->assertMatchesJsonSnapshot($res3->getBody());
        $this->assertEquals(200, $res3->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res3->getHeader("Content-Type"));

    }

    public function testDiscountCode1(): void {

        /*
         *  Step 1: Add product "test-simple" to quote
         */
        MagentoManager::reset();
        MagentoManager::init();

        $product1 = Mage::getModel("catalog/product");
        $product1->load($product1->getIdBySku("test-simple"));
        $buyRequest1 = json_encode(new MageQL_Sales_Model_BuyRequest_Product($product1));

        $res1 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            addQuoteItem(buyRequest:' . $buyRequest1 . ',qty:1) { result }
        }'));

        $this->assertMatchesJsonSnapshot($res1->getBody());
        $this->assertEquals(200, $res1->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res1->getHeader("Content-Type"));

        /*
         *  Step 2: Apply discount code "PHPUnitTest1"
         */
        self::resetAndInitButKeepQuote();
        $res2 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            setQuoteDiscountCode(code:"PHPUnitTest1") { result, coupon { code, label } }
        }'));

        $this->assertMatchesJsonSnapshot($res2->getBody());
        $this->assertEquals(200, $res2->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res2->getHeader("Content-Type"));

        /*
         *  Step 3: Add product "test-virtual" to quote
         */
        self::resetAndInitButKeepQuote();
        $product2 = Mage::getModel("catalog/product");
        $product2->load($product2->getIdBySku("test-virtual"));
        $buyRequest2 = json_encode(new MageQL_Sales_Model_BuyRequest_Product($product2));

        $res3 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            addQuoteItem(buyRequest:' . $buyRequest2 . ',qty:1) { result }
        }'));

        $this->assertMatchesJsonSnapshot($res3->getBody());
        $this->assertEquals(200, $res3->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res3->getHeader("Content-Type"));

        /*
         *  Step 4: Fetch quote info
         *  Make sure discount is applied to both products
         */
        self::resetAndInitButKeepQuote();
        $res4 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            quote {
                items { product { sku, price { incVat } } }
                coupon { code, label }
                discounts { label, type }
                grandTotal { incVat }
                subTotal { incVat }
                discountTotal
            }
        }'));

        $this->assertMatchesJsonSnapshot($res4->getBody());
        $this->assertEquals(200, $res4->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res4->getHeader("Content-Type"));

        /*
         *  Step 5: Find "itemBuyRequest" for product "test-simple" in quote
         */
        self::resetAndInitButKeepQuote();
        $res5 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            quote { items { itemBuyRequest, product { sku } } }
        }'));

        $quoteData = json_decode($res5->getBody(), true);
        $itemBuyRequest = "";
        if (!empty($quoteData["data"]["quote"]["items"])) {
            foreach ($quoteData["data"]["quote"]["items"] as $quoteItem) {
                if (!empty($quoteItem["itemBuyRequest"]) && !empty($quoteItem["product"]["sku"])) {
                    if ($quoteItem["product"]["sku"] == "test-simple") {
                        $itemBuyRequest = json_encode($quoteItem["itemBuyRequest"]);
                    }
                }
            }
        }
        $this->assertNotEmpty($itemBuyRequest);

        /*
         *  Step 6: Remove product "test-simple" from quote
         */
        self::resetAndInitButKeepQuote();
        $res6 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            removeQuoteItem(itemBuyRequest:' . $itemBuyRequest . ') { result }
        }'));

        $this->assertMatchesJsonSnapshot($res6->getBody());
        $this->assertEquals(200, $res6->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res6->getHeader("Content-Type"));

        /*
         *  Step 7: Fetch quote info
         *  Make sure discount still is applied to remaining product
         */
        self::resetAndInitButKeepQuote();
        $res7 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            quote {
                items { product { sku, price { incVat } } }
                coupon { code, label }
                discounts { label, type }
                grandTotal { incVat }
                subTotal { incVat }
                discountTotal
            }
        }'));

        $this->assertMatchesJsonSnapshot($res7->getBody());
        $this->assertEquals(200, $res7->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res7->getHeader("Content-Type"));

        /*
         *  Step 8: Remove discount code
         */
        self::resetAndInitButKeepQuote();
        $res8 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            removeQuoteDiscountCode { result }
        }'));

        $this->assertMatchesJsonSnapshot($res8->getBody());
        $this->assertEquals(200, $res8->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res8->getHeader("Content-Type"));

        /*
         *  Step 9: Fetch quote info
         *  Make sure no discount is applied to remaining product
         */
        self::resetAndInitButKeepQuote();
        $res9 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            quote {
                items { product { sku, price { incVat } } }
                coupon { code, label }
                discounts { label, type }
                grandTotal { incVat }
                subTotal { incVat }
                discountTotal
            }
        }'));

        $this->assertMatchesJsonSnapshot($res9->getBody());
        $this->assertEquals(200, $res9->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res9->getHeader("Content-Type"));

    }

    public function testDiscountCode2(): void {

        /*
         *  Step 1: Add product "test-simple" to quote
         */
        MagentoManager::reset();
        MagentoManager::init();
        $product1 = Mage::getModel("catalog/product");
        $product1->load($product1->getIdBySku("test-simple"));
        $buyRequest1 = json_encode(new MageQL_Sales_Model_BuyRequest_Product($product1));

        $res1 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            addQuoteItem(buyRequest:' . $buyRequest1 . ',qty:1) { result }
        }'));

        $this->assertMatchesJsonSnapshot($res1->getBody());
        $this->assertEquals(200, $res1->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res1->getHeader("Content-Type"));

        /*
         *  Step 2: Apply discount code "PHPUnitTest2"
         *  Make sure it fails with "errorValidCodeNotUsable"
         */
        self::resetAndInitButKeepQuote();
        $res2 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            setQuoteDiscountCode(code:"PHPUnitTest2") { result, coupon { code, label } }
        }'));

        $this->assertMatchesJsonSnapshot($res2->getBody());
        $this->assertEquals(200, $res2->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res2->getHeader("Content-Type"));

        /*
         *  Step 3: Add product "test-virtual" to quote
         */
        self::resetAndInitButKeepQuote();
        $product2 = Mage::getModel("catalog/product");
        $product2->load($product2->getIdBySku("test-virtual"));
        $buyRequest2 = json_encode(new MageQL_Sales_Model_BuyRequest_Product($product2));

        $res3 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            addQuoteItem(buyRequest:' . $buyRequest2 . ',qty:1) { result }
        }'));

        $this->assertMatchesJsonSnapshot($res3->getBody());
        $this->assertEquals(200, $res3->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res3->getHeader("Content-Type"));

        /*
         *  Step 4: Apply discount code "PHPUnitTest2"
         */
        self::resetAndInitButKeepQuote();
        $res4 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            setQuoteDiscountCode(code:"PHPUnitTest2") { result, coupon { code, label } }
        }'));

        $this->assertMatchesJsonSnapshot($res4->getBody());
        $this->assertEquals(200, $res4->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res4->getHeader("Content-Type"));

        /*
         *  Step 5: Fetch quote info
         *  Make sure discount is applied
         */
        self::resetAndInitButKeepQuote();
        $res5 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            quote {
                items { product { sku, price { incVat } } }
                coupon { code, label }
                discounts { label, type }
                grandTotal { incVat }
                subTotal { incVat }
                discountTotal
            }
        }'));

        $this->assertMatchesJsonSnapshot($res5->getBody());
        $this->assertEquals(200, $res5->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res5->getHeader("Content-Type"));

        /*
         *  Step 6: Find "itemBuyRequest" for product "test-virtual" in quote
         */
        self::resetAndInitButKeepQuote();
        $res6 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            quote { items { itemBuyRequest, product { sku } } }
        }'));

        $quoteData = json_decode($res6->getBody(), true);
        $itemBuyRequest = "";
        if (!empty($quoteData["data"]["quote"]["items"])) {
            foreach ($quoteData["data"]["quote"]["items"] as $quoteItem) {
                if (!empty($quoteItem["itemBuyRequest"]) && !empty($quoteItem["product"]["sku"])) {
                    if ($quoteItem["product"]["sku"] == "test-virtual") {
                        $itemBuyRequest = json_encode($quoteItem["itemBuyRequest"]);
                    }
                }
            }
        }
        $this->assertNotEmpty($itemBuyRequest);

        /*
         *  Step 7: Remove product "test-virtual" from quote
         */
        self::resetAndInitButKeepQuote();
        $res7 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            removeQuoteItem(itemBuyRequest:' . $itemBuyRequest . ') { result }
        }'));

        $this->assertMatchesJsonSnapshot($res7->getBody());
        $this->assertEquals(200, $res7->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res7->getHeader("Content-Type"));

        /*
         *  Step 8: Fetch quote info
         *  Make sure discount is no longer applied
         */
        self::resetAndInitButKeepQuote();
        $res8 = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            quote {
                items { product { sku, price { incVat } } }
                coupon { code, label }
                discounts { label, type }
                grandTotal { incVat }
                subTotal { incVat }
                discountTotal
            }
        }'));

        $this->assertMatchesJsonSnapshot($res8->getBody());
        $this->assertEquals(200, $res8->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res8->getHeader("Content-Type"));

    }

}
