<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;
use function MageQL\snakeCaseToCamel;

class MageQL_Sales_Model_Discount {
    const SUCCESS = "success";
    const ERROR_SETTING_CODE = "errorSettingDiscountCode";
    const ERROR_REMOVING_CODE = "errorRemovingDiscountCode";
    const ERROR_CODE_NOT_FOUND = "errorDiscountCodeNotFound";
    const ERROR_CODE_EMPTY = "errorDiscountCodeEmpty";
    const ERROR_VALID_CODE_NOT_USABLE = "errorValidCodeNotUsable";

    const DISCOUNT_TYPE_NO_COUPON = "discountTypeNoCoupon";
    const DISCOUNT_TYPE_SPECIFIC = "discountTypeSpecificCoupon";
    const DISCOUNT_TYPE_AUTO = "discountTypeMultipleCoupons";

    /**
     * @param mixed $unusedSrc
     * @return ?array{code: string, label: string}
     */
    public static function resolveDiscountCode($unusedSrc, array $unusedArgs, MageQL_Core_Model_Context $unusedCtx) {
        $quote = Mage::getSingleton("mageql_sales/quote")->getQuote();

        if( ! $quote->getCouponCode()) {
            return null;
        }

        $coupon = Mage::getModel("salesrule/coupon")->loadByCode($quote->getCouponCode());

        if(!$coupon->getRuleId()) {
            return null;
        }

        $rule = Mage::getModel("salesrule/rule")->load($coupon->getRuleId());

        return [
            "code" => $quote->getCouponCode(),
            "label"      => (string)$rule->getStoreLabel(),
        ];
    }

    /**
     * @param mixed $unusedSrc
     * @return ?array
     */
    public static function resolveDiscounts(
        $unusedSrc,
        array $unusedArgs,
        MageQL_Core_Model_Context $unusedCtx
    ) {
        $quote = Mage::getSingleton("mageql_sales/quote")->getQuote();
        $collectedRules = [];

        foreach ($quote->getAllVisibleItems() as $item) {
            if ($item->getAppliedRuleIds()) {
                $itemRuleIds = explode(',', $item->getAppliedRuleIds());

                foreach ($itemRuleIds as $ruleId) {
                    if (empty($collectedRules[$ruleId])) {
                        $rule = Mage::getModel('salesrule/rule')->load((int)$ruleId);

                        $collectedRules[$ruleId] = [
                            "label" => $rule->getStoreLabel(),
                            "type" => $rule->getCouponType(),
                        ];
                    }
                }
            }
        }

        return !empty($collectedRules) ? $collectedRules : null;
    }

    public static function resolveDiscountTotal(Mage_Sales_Model_Quote $quote): float {
        $totals = $quote->getTotals();

        return array_key_exists("discount", $totals) ? (double) $totals["discount"]->getValue() : 0;
    }

    /**
     * Add discount code
     *
     * @param mixed $unusedSrc
     * @return string
     */
    public static function mutateSetQuoteDiscountCode($unusedSrc, array $args, MageQL_Core_Model_Context $unusedCtx) {
        try {
            if (empty($args["code"])) {
                return self::ERROR_CODE_EMPTY;
            }

            $code = $args["code"];
            $coupon = Mage::getModel('salesrule/coupon')->loadByCode($code);
            $data = $coupon->getData();

            if (empty($data)) {
                return self::ERROR_CODE_NOT_FOUND;
            }

            $model = Mage::getSingleton("mageql_sales/quote");
            $quote = $model->getQuote();

            $quote->setCouponCode($code);
            $model->saveSessionQuoteWithShippingRates();

            $newCode = $quote->getCouponCode();

            if (empty($newCode)) {
                return self::ERROR_VALID_CODE_NOT_USABLE;
            }

            return self::SUCCESS;
        }
        catch (Exception $ex) {
            Mage::logException($ex);

            return self::ERROR_SETTING_CODE;
        }
    }

    /**
     * Remove discount code
     *
     * @param mixed $unusedSrc
     * @return string
     */
    public static function mutateRemoveQuoteDiscountCode($unusedSrc, array $unusedArgs, MageQL_Core_Model_Context $unusedCtx) {
        try {
            $model = Mage::getSingleton("mageql_sales/quote");
            $quote = $model->getQuote();

            $quote->setCouponCode("");
            $model->saveSessionQuoteWithShippingRates();

            return self::SUCCESS;
        }
        catch (Exception $ex) {
            Mage::logException($ex);

            return self::ERROR_REMOVING_CODE;
        }
    }
}
