<?php

declare(strict_types=1);

use function MageQL\snakeCaseToCamel;

class MageQL_Sales_Model_Shipping extends Mage_Core_Model_Abstract {
    /**
     * @return Array<Mage_Sales_Model_Quote_Address_Rate>
     */
    public static function resolveAll(): array {
        $quoteModel = Mage::getSingleton("mageql_sales/quote");
        $quote = $quoteModel->getQuote();

        if( ! $quote->hasItems()) {
            return [];
        }

        $shipping = $quoteModel->getShippingAddress();

        if( ! $shipping) {
            return [];
        }

        $shippingMethods = [];
        // We can't fetch this using getShippingRatesCollection since then they
        // will not be sorted.
        $carriers = $shipping->setCollectShippingRates(true)
             ->collectShippingRates()
             ->getGroupedAllShippingRates();

        foreach($carriers as $methods) {
            foreach($methods as $method) {
                $shippingMethods[] = $method;
            }
        }

        return $shippingMethods;
    }

    public static function resolvePriceExVat(
        Mage_Sales_Model_Quote_Address_Rate $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): float {
        $tax = Mage::helper("tax");

        if( ! $tax->shippingPriceIncludesTax()) {
            return $src->getPrice();
        }

        $quoteModel = Mage::getSingleton("mageql_sales/quote");
        $shippingAddress = $quoteModel->getShippingAddress();
        $billingAddress = $quoteModel->getBillingAddress();
        // FIXME: Customer tax class
        $customerTaxClass = null;

        if( ! $shippingAddress || ! $billingAddress) {
            throw new Exception(sprintf(
                "%s: Attempting to calculate shipping price without any valid address or quote",
                __METHOD__
            ));
        }

        return $tax->getShippingPrice(
            $src->getPrice(),
            false,
            $shippingAddress,
            $customerTaxClass,
            $ctx->getStore()
        );
    }

    public static function resolvePriceIncVat(
        Mage_Sales_Model_Quote_Address_Rate $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): float {
        $tax = Mage::helper("tax");

        if($tax->shippingPriceIncludesTax()) {
            return $src->getPrice();
        }

        $quoteModel = Mage::getSingleton("mageql_sales/quote");
        $shippingAddress = $quoteModel->getShippingAddress();
        $billingAddress = $quoteModel->getBillingAddress();
        // FIXME: Customer tax class
        $customerTaxClass = null;

        if( ! $shippingAddress || ! $billingAddress) {
            throw new Exception(sprintf(
                "%s: Attempting to calculate shipping price without any valid address or quote",
                __METHOD__
            ));
        }

        return $tax->getShippingPrice(
            $src->getPrice(),
            true,
            $shippingAddress,
            $customerTaxClass,
            $ctx->getStore()
        );
    }

    public static function resolvePriceVat(
        Mage_Sales_Model_Quote_Address_Rate $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): float {
        $tax = Mage::helper("tax");
        $quoteModel = Mage::getSingleton("mageql_sales/quote");
        $store = $ctx->getStore();
        $shippingAddress = $quoteModel->getShippingAddress();
        $billingAddress = $quoteModel->getBillingAddress();
        // FIXME: Customer tax class
        $customerTaxClass = null;

        if( ! $shippingAddress || ! $billingAddress) {
            throw new Exception(sprintf(
                "%s: Attempting to calculate shipping price without any valid address or quote",
                __METHOD__
            ));
        }

        if($tax->shippingPriceIncludesTax()) {
            return $store->roundPrice($src->getPrice() - $tax->getShippingPrice(
                $src->getPrice(),
                false,
                $shippingAddress,
                $customerTaxClass,
                $store
            ));
        }

        return $store->roundPrice($tax->getShippingPrice(
            $src->getPrice(),
            true,
            $shippingAddress,
            $customerTaxClass,
            $store
        ) - $src->getPrice());
    }
}
