<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use MageQL\Type\InputFieldBuilder;

class MageQL_Sales_Model_Schema_Default_Order extends MageQL_Core_Model_Schema_Abstract {
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "Order":
            return $this->object("A submitted order")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "OrderAddress":
            return $this->object("An address of a submitted order")
                ->setInterfaces(["Address"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "OrderItem":
            return $this->interface("An item in the placed order")
                ->setTypeResolver(function(Mage_Sales_Model_Order_Item $src) {
                    switch($src->getProduct()->getTypeId()) {
                    case "configurable":
                        return "OrderItemConfigurable";
                    case "bundle":
                        return "OrderItemBundle";
                    case "simple":
                    case "virtual":
                        return "OrderItemSimple";
                    default:
                        throw new Exception(sprintf(
                            "Unknown product type-id %s for quote item",
                            $src->getProduct()->getTypeId()
                        ));
                    }
                });

        case "OrderItemBundle":
            return $this->object("An item in the order with bundle selections applied")
                ->setInterfaces(["OrderItem"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "OrderItemConfigurable":
            return $this->object("An item in the order with a configuration option applied")
                ->setInterfaces(["OrderItem"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "OrderItemSimple":
            return $this->object("An item in the order without any configuration options")
                ->setInterfaces(["OrderItem"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "OrderShipping":
            return $this->object("Used shipping for the order");

        case "OrderShippingMethod":
            return $this->object("Order shipping mehtod")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "OrderStatus":
            $orderConfig = Mage::getSingleton("sales/order_config");

            return $this->enum("Order status", array_map(function($v) {
                return [
                    "description" => $v,
                ];
            }, $orderConfig->getStatuses()));

        case "OrderTotal":
            return $this->object("A total in the order")
                ->setInterfaces(["Price"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "PaginatedOrders":
            return $this->object("Paginated list of orders, sorted chronologically by default.");
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "Customer":
            return [
                "orders" => $this->field("PaginatedOrders!", "List of previously placed orders for the customer")
                    ->addArgument(
                        "pageSize",
                        $this->argument("Int", "Maximum number of orders to list")
                            ->setDefaultValue(20))
                    ->addArgument(
                        "page",
                        $this->argument("Int", "Which page to show")
                            ->setDefaultValue(1))
                    ->setResolver("MageQL_Sales_Model_Customer::resolveOrders"),
            ];

        case "Order":
            return [
                "status" => $this->field("OrderStatus!", "Order status")
                    ->setResolver("MageQL_Sales_Model_Order::resolveStatus"),
                "addresses" => $this->field("[OrderAddress!]!", "List of addresses related to this quote")
                    ->setResolver("MageQL_Sales_Model_Order::resolveAddresses"),
                "currencyCode" => $this->field("String!", "Currency code used by the order")
                    ->setResolver("MageQL_Sales_Model_Order::resolveCurrencyCode"),
                "email" => $this->field("String!", "The customer email for the order")
                    ->setResolver("MageQL_Sales_Model_Order::resolveEmail"),
                "grandTotal" => $this->field("OrderTotal!", "Order total")
                    ->setResolver("MageQL_Sales_Model_Order::resolveGrandTotal"),
                "id" => $this->field("ID!", "The order id")
                    ->setResolver("MageQL_Sales_Model_Order::resolveId"),
                "items" => $this->field("[OrderItem!]!", "The ordered items")
                    ->setResolver("MageQL_Sales_Model_Order::resolveItems"),
                "createdAt" => $this->field("String!", "Order datetime, ISO 8601")
                    ->setResolver("MageQL_Sales_Model_Order::resolveCreatedAt"),
                "shipping" => $this->field("OrderShipping", "Shipping information for the order, null if virtual")
                    ->setResolver("MageQL_Sales_Model_Order::resolveShipping"),
                "subTotal" => $this->field("OrderTotal!", "Order subtotal")
                    ->setResolver("MageQL_Sales_Model_Order::resolveSubTotal"),
                "discountTotal" => $this->field("Float!", "Total of applied discounts")
                    ->setResolver("MageQL_Sales_Model_Order::resolveDiscountTotal"),
                "virtual" => $this->field("Boolean!", "If the order only contains virtual products")
                    ->setResolver("MageQL_Sales_Model_Order::resolveVirtual"),
            ];

        case "OrderAddress":
            return array_merge([
                "type" => $this->field("QuoteAddressType!", "Type of address")
                    ->setResolver("MageQL_Sales_Model_Quote_Address::resolveType"),
            ], $registry->getFieldBuilders("Address"));

        case "OrderItem":
            return [
                "qty" => $this->field("Int!", "Quantity of this product item in the order")
                    ->setResolver("MageQL_Sales_Model_Order_Item::resolveQty"),
                "rowTotal" => $this->field("ItemRowTotal!", "Row total")
                    ->setResolver("MageQL\\forwardResolver"),
                // TODO: Should we use this? This is the actual SKU (aka. child-product sku for config)
                // "sku" => $this->field("String!", "SKU of the selected product"),
                "product" => $this->field("ListProduct!", "The selected product")
                    ->setResolver("MageQL_Sales_Model_Item::resolveProduct"),
            ];

        case "OrderItemBundle":
            return array_merge($registry->getFieldBuilders("OrderItem"), [
                "bundleOptions" => $this->field("[ItemBundleOption!]!", "The selected product options")
                    ->setResolver("MageQL_Sales_Model_Item::resolveOrderBundleOptions"),
            ]);

        case "OrderItemConfigurable":
            return array_merge($registry->getFieldBuilders("OrderItem"), [
                "configOption" => $this->field("ItemConfigurableOption!", "The selected product option")
                    ->setResolver("MageQL_Sales_Model_Item::resolveOrderConfigOption"),
            ]);

        case "OrderItemSimple":
            return $registry->getFieldBuilders("OrderItem");

        case "OrderShipping":
            return [
                "method" => $this->field("OrderShippingMethod!", "Selected method for shipping")
                    ->setResolver("MageQL_Sales_Model_Order_Shipping::resolveMethod"),
                "total" => $this->field("OrderTotal!", "Total for shipping")
                    ->setResolver("MageQL_Sales_Model_Order_Shipping::resolveTotal"),
            ];

        case "OrderShippingMethod":
            return [
                "code" => $this->field("ID!", "Shipping code"),
                "description" => $this->field("String!", "Shipping method description"),
            ];

        case "OrderTotal":
            return [
                "exVat" => $this->field("Float!", "Price excluding VAT"),
                "incVat" => $this->field("Float!", "Price including VAT"),
                "vat" => $this->field("Float!", "VAT amount"),
            ];

        case "PaginatedOrders":
            return [
                "items" => $this->field("[Order!]!", "List of orders")
                    ->setResolver("MageQL_Sales_Model_Customer::resolvePaginatedItems"),
                "totalCount" => $this->field("Int!", "Total number of orders in this paginated collection")
                    ->setResolver(function(Varien_Data_Collection $collection) {
                        return $collection->getSize();
                    }),
            ];

        case "Query":
            return [
                // TODO: "checkoutAgreements" => $this->field("[CheckoutAgreement!]!", "List of agreements, some of which are mandatory")
                "lastOrder" => $this->field("Order", "The last submitted order if any")
                    ->setResolver("MageQL_Sales_Model_Order::resolveLast"),
                "orderById" => $this->field("Order", "A specific order by increment id, will result in null if order is not found, if the customer is not logged in, or if the customer is not the creator of the order.")
                    ->addArgument("id", $this->argument("ID!", "Order increment id"))
                    ->setResolver("MageQL_Sales_Model_Customer::resolveOrderById"),
            ];
        }

        return [];
    }

    public function getUnreachableTypes(): array {
        return [
            "OrderItemBundle",
            "OrderItemConfigurable",
            "OrderItemSimple",
        ];
    }
}
