<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Sales_Model_Payment {
    /**
     * @var ?Array<MageQL_Sales_Model_Payment_Abstract>
     */
    protected $models = null;

    /**
     * Returns a list of enabled payment methods for the current store.
     *
     * @return Array<MageQL_Sales_Model_Payment_Abstract>
     */
    public function getModels(): array {
        if($this->models === null) {
            $this->models = [];

            $store = Mage::app()->getStore();
            /**
             * @var Array<string, array{model:string}>
             */
            $payments = $store->getConfig("mageql/sales/payment");

            foreach($payments as $code => $data) {
                /**
                 * @var MageQL_Sales_Model_Payment_Abstract|null
                 */
                $model = Mage::getSingleton($data["model"], [
                    "context" => Mage::registry(MageQL_Core_Model_Context::REGISTRY_KEY),
                ]);

                if( ! $model instanceof MageQL_Sales_Model_Payment_Abstract) {
                    throw new Exception(sprintf(
                        "%s: MageQL sales payment model '%s' implementing %s for method with code '%s' could not be found",
                        __METHOD__,
                        $data["model"],
                        MageQL_Sales_Model_Payment_Abstract::class,
                        $code
                    ));
                }

                $code = $model->getCode();

                // We do not want to include inactives since that will give the
                // wrong impression to clients if types for unsupported payment
                // types exist
                if($store->getConfig("payment/$code/active")) {
                    $this->models[$code] = $model;
                }
            }
        }

        return $this->models;
    }

    /**
     * Returns true if this schema has support for the supplied payment method.
     */
    public function hasPaymentMethod(string $code): bool {
        foreach($this->getModels() as $m) {
            if($m->getCode() === $code) {
                return true;
            }
        }

        return false;
    }

    public function typePaymentMethod(Mage_Payment_Model_Method_Abstract $src): string {
        $match = $this->getModels()[$src->getCode()] ?? null;

        if( ! $match) {
            // This should not happen unless paymentMethods resolver is broken
            throw new Exception(sprintf("%s: Unsupported payment method '%s'", __METHOD__, $src->getCode()));
        }

        return $match->getPaymentMethodType();
    }

    public function typeQuotePayment(Mage_Sales_Model_Quote_Payment $src): string {
        $match = $this->getModels()[$src->getMethod()] ?? null;

        if( ! $match) {
            // This should not happen unless it is a broken quote, or the quote payment resolver is broken
            throw new Exception(sprintf("%s: Unsupported payment method '%s'", __METHOD__, $src->getMethod()));
        }

        return $match->getQuotePaymentType();
    }

    public function resolvePaymentMethods(): array {
        $methods = [];
        $paymentHelper = Mage::helper("payment");
        $quote = Mage::getSingleton("mageql_sales/quote")->getQuote();
        $payment = $quote->getPayment();

        // This function uses isAvailable on the instance
        foreach($paymentHelper->getStoreMethods($quote->getStore(), $quote) as $method) {
            if($method->canUseCheckout()) {
                foreach($this->getModels() as $m) {
                    if($m->getCode() === $method->getCode()) {
                        $method->setInfoInstance($payment);

                        $methods[] = $method;
                    }
                }
            }
        }

        return $methods;
    }

    /**
     * @return Array<string, array{errorCode?:int, description:string}>
     */
    public function getPlaceOrderErrors(): array {
        $models = $this->getModels();
        $errors = [];

        foreach($models as $m) {
            $errors = array_merge($errors, $m->getPlaceOrderErrors());
        }

        return $errors;
    }
}
