<?php

declare(strict_types=1);

namespace MageQL\Sales;

use Mage;
use Mage_Customer_Model_Customer;
use MageQL_Sales_Model_BuyRequest;
use Throwable;
use Varien_Object;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class QuoteTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();

        $this->clearQuotes();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        $this->clearQuotes();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        $this->clearQuotes();

        throw $e;
    }

    protected function clearQuotes(): void {
        MagentoManager::init();

        // Prevent any existing quotes from being loaded
        $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

        $conn->query("UPDATE sales_flat_quote SET is_active = 0");

        MagentoManager::reset();
    }

    protected function customerFixture(): Mage_Customer_Model_Customer {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        return Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");
    }

    public function testRemoveQuoteEmpty(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    email
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        product {
                            sku
                            attributes {
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                    }
                    shipping {
                        method {
                            description
                        }
                    }
                    validationErrors {
                        code
                        message
                    }
                }
            }'],
            [ "query" => 'mutation {
                removeQuote
            }'],
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    email
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        product {
                            sku
                            attributes {
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                    }
                    shipping {
                        method {
                            description
                        }
                    }
                    validationErrors {
                        code
                        message
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveQuote(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    email
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        product {
                            sku
                            attributes {
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                    }
                    shipping {
                        method {
                            description
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                removeQuote
            }'],
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    email
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        product {
                            sku
                            attributes {
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                    }
                    shipping {
                        method {
                            description
                        }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testQuoteTaxRates(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                taxRates {
                    percent
                    amount
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testEmail(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetEmail(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteEmail(email: "test@example.com")  {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteEmail(email: "another@example.com")  {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuote()->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                email
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetEmailInvalid(): void {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteEmail(email: "invalid")  {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetEmailLoggedIn(): void {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteEmail(email: "invalid")  {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoggedInAddress(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // Prevent any existing quotes from being loaded
        $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

        $conn->query("UPDATE sales_flat_quote SET is_active = 0");

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $session = Mage::getSingleton("customer/session");

        $session->login("test-customer@example.com", "test-customer");
        $customer->setSession($session->getSessionId());

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoggedInAddressAlreadySet(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        $customer = $this->customerFixture();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $session = Mage::getSingleton("customer/session");

        $session->login("test-customer@example.com", "test-customer");
        $customer->setSession($session->getSessionId());
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $quote->getShippingAddress()->addData([
            "firstname" => "First",
            "lastname" => "New ",
            "city" => "The new city",
            "country_id" => "SE",
            "postcode" => "51555",
            "street" => "The new street shipping",
            "telephone" => "6712358234",
        ]);

        $quote->getBillingAddress()->addData([
            "firstname" => "T",
            "lastname" => "Testsson ",
            "city" => "The new city",
            "country_id" => "SE",
            "postcode" => "51555",
            "street" => "The new street shipping",
            "telephone" => "6712358234",
        ]);

        $quote->save();

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoggedInAddressAfterAddQuoteItem(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $this->customerFixture();

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            loginCustomer(email: "test-customer@example.com", password: "test-customer") {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetQuoteAddressesToCustomerDefaults(): void {
        MagentoManager::init();

        // We need a product to ensure the shipping address is set
        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteAddressesToCustomerDefaults {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetQuoteAddressesToCustomerDefaultsNotLoggedIn(): void {
        MagentoManager::init();

        // We need a product to ensure the shipping address is set
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteAddressesToCustomerDefaults  {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetQuoteAddressesToCustomerDefaultsBilling(): void {
        MagentoManager::init();

        // We need a product to ensure the shipping address is set
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "firstname" => "Billing",
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "firstname" => "Shipping",
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteAddressesToCustomerDefaults(billing: true, shipping: false) {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetQuoteAddressesToCustomerDefaultsShipping(): void {
        MagentoManager::init();

        // We need a product to ensure the shipping address is set
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "firstname" => "Billing",
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "firstname" => "Shipping",
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuoteAddressesToCustomerDefaults(billing: false, shipping: true) {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testZeroGrandTotalWithTax(): void {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Zero grand total but positive tax, can be caused by things like discounts
        $quote->setGrandTotal(0);
        $quote->getShippingAddress()->setTaxAmount(123);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                subTotal {
                    incVat
                }
                discountTotal
                grandTotal {
                    incVat
                    exVat
                    vat
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
