<?php

declare(strict_types=1);

namespace MageQL\Sales\Wishlist;

use Mage;
use Throwable;
use Varien_Object;

use Crossroads\Magento\Test\Integration\MagentoManager;
use PHPUnit\Framework\TestCase;

class Base extends TestCase {
    public function setUp(): void {
        MagentoManager::reset();
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $storeId = $store->getId();

        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");
        $customerId = $customer->getId();

        $wishlist = Mage::getModel('wishlist/wishlist')->loadByCustomer($customerId, true);

        foreach($wishlist->getItemCollection() as $item) {
            $item->delete();
        }

        $wishlist = Mage::getModel('wishlist/wishlist')->loadByCustomer($customerId, true);

        $simple = Mage::getModel("catalog/product");
        $simpleId = $simple->getIdBySku("test-simple");
        $simple->setStoreId($storeId)->load($simpleId);

        $buyRequestSimple = [
            "product" => $simpleId,
            "qty" => 1
        ];

        $simpleItem = $wishlist->addNewItem($simple, new Varien_Object($buyRequestSimple));

        $this->assertIsObject($simpleItem);

        $simpleItem->setAddedAt("2020-01-18 17:00:00");

        $config = Mage::getModel("catalog/product");
        $configId = $config->getIdBySku("test-config");
        $config->setStoreId($storeId)->load($configId);

        $attribute = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");

        $this->assertNotFalse($attribute);

        $attributeId = $attribute->getId();
        $attributeOptions = $attribute->getSource()->getAllOptions();
        $attributeRed = null;

        foreach($attributeOptions as $o) {
            switch($o["label"]) {
            case "Red":
                $attributeRed = $o["value"];
                break;
            }
        }
        $buyRequestConfig = [
            "product" => $configId,
            "qty" => 1,
            "super_attribute" => [
                $attributeId => $attributeRed
            ]
        ];

        $configItem = $wishlist->addNewItem($config, new Varien_Object($buyRequestConfig));

        $this->assertIsObject($configItem);

        $configItem->setAddedAt("2020-01-01 17:00:00");

        $wishlist->save();

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function fetchWishlistItemId(string $sku): ?string {
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $storeId = $store->getId();

        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");
        $customerId = $customer->getId();

        $sqlQuery = "
            SELECT
                wi.wishlist_item_id
            FROM catalog_product_entity cpe
            JOIN wishlist_item wi ON wi.product_id = cpe.entity_id
            JOIN wishlist w ON w.wishlist_id = wi.wishlist_id
            WHERE
                cpe.sku = :sku
                AND wi.store_id = :storeId
                AND w.customer_id = :customerId
        ";
        $params = [
            "sku" => $sku,
            "storeId" => $storeId,
            "customerId" => $customerId
        ];

        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params);
    }
}
