<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

class MageQL_Sales_Model_Quote_Shipping extends Mage_Core_Model_Abstract {
    const SUCCESS = "success";
    const NOT_MODIFIED = "notModified";
    const NO_ITEMS = "errorNoItems";
    const NOT_FOUND = "errorNotFound";

    public static function resolveMethod(Mage_Sales_Model_Quote_Address $src): string {
        return $src->getShippingMethod();
    }

    public static function resolveDescription(Mage_Sales_Model_Quote_Address $src): string {
        return $src->getShippingDescription();
    }

    public static function resolvePriceExVat(Mage_Sales_Model_Quote_Address $src): float {
        return (float)$src->getShippingAmount();
    }

    public static function resolvePriceIncVat(Mage_Sales_Model_Quote_Address $src): float {
        return (float)$src->getShippingInclTax();
    }

    public static function resolvePriceVat(Mage_Sales_Model_Quote_Address $src): float {
        return (float)$src->getShippingTaxAmount();
    }

    /**
     * @param mixed $unusedSrc
     * @param array{overwrite:bool} $args
     */
    public static function mutateSetToCheapest($unusedSrc, array $args): string {
        $model = Mage::getSingleton("mageql_sales/quote");
        $quote = $model->getQuote();
        $overwrite = $args["overwrite"] ?? false;
        $address = $model->getShippingAddress();

        if( ! $address || ! $quote->hasItems()) {
            return self::NO_ITEMS;
        }

        if( ! $overwrite && $address->getShippingMethod()) {
            return self::NOT_MODIFIED;
        }

        // We can't fetch this using getShippingRatesCollection since then they
        // will not be sorted.
        $carriers = $address->setCollectShippingRates(true)
             ->collectShippingRates()
             ->getGroupedAllShippingRates();
        $cheapest = null;

        foreach($carriers as $methods) {
            foreach($methods as $method) {
                // We go through them in order, in case we get two with the same
                // price it will pick the first one
                if( ! $cheapest || $cheapest->getPrice() > $method->getPrice()) {
                    $cheapest = $method;
                }
            }
        }

        if( ! $cheapest) {
            return self::NOT_FOUND;
        }

        return self::setShippingMethod($address, $cheapest);
    }

    /**
     * @param mixed $unusedSrc
     * @param array{shippingMethodCode:string} $args
     */
    public static function mutateSet($unusedSrc, array $args): string {
        $model = Mage::getSingleton("mageql_sales/quote");
        $quote = $model->getQuote();
        $address = $model->getShippingAddress();
        $code = $args["shippingMethodCode"];

        if( ! $address || ! $quote->hasItems()) {
            return self::NO_ITEMS;
        }

        if($address->getShippingMethod() === $code) {
            return self::NOT_MODIFIED;
        }

        $rate = $address->getShippingRateByCode($code);

        if( ! $rate) {
            return self::NOT_FOUND;
        }

        return self::setShippingMethod($address, $rate);
    }

    public static function setShippingMethod(
        Mage_Sales_Model_Quote_Address $address,
        Mage_Shipping_Model_Rate_Abstract $method
    ): string {
        if($address->getShippingMethod() === $method->getCode()) {
            return self::NOT_MODIFIED;
        }

        $address->setShippingMethod($method->getCode());
        $address->setCollectShippingRates(true);
        $address->getQuote()->setTotalsCollectedFlag(false);

        Mage::getSingleton("mageql_sales/quote")->saveSessionQuote();

        return self::SUCCESS;
    }
}
