<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Sales_Model_Quote_Address extends Mage_Core_Model_Abstract {
    const COPY_FIELDS = [
        "prefix",
        "firstname",
        "middlename",
        "lastname",
        "suffix",
        "company",
        "street",
        "city",
        "postcode",
        "region",
        "region_id",
        "country_id",
        "telephone",
    ];

    /**
     * Fields which are part of an address.
     */
    const ADDRESS_DATA_FIELDS = [
        "prefix",
        "firstname",
        "middlename",
        "lastname",
        "suffix",
        "company",
        "street",
        "city",
        "postcode",
        "region",
        "region_id",
        "telephone",
        "country_id",
    ];

    public static function resolveIsUsedAsShipping(Mage_Customer_Model_Address_Abstract $src): bool {
        if($src->getAddressType() === Mage_Sales_Model_Quote_Address::TYPE_SHIPPING) {
            return true;
        }

        // TODO: Use the quote from $src?
        $shipping = Mage::getSingleton("mageql_sales/quote")->getShippingAddress();

        return $shipping ? (bool)$shipping->getSameAsBilling() : false;
    }

    public static function resolveType(Mage_Customer_Model_Address_Abstract $src): string {
        return $src->getAddressType();
    }

    /**
     * @param mixed $unusedSrc
     * @param array{billing:bool, shipping:bool} $args
     * @return MageQL_Sales_Model_Quote_Address_Result::SUCCESS|MageQL_Sales_Model_Quote_Address_Result::NOT_MODIFIED|MageQL_Sales_Model_Quote_Address_Result::ERROR_NOT_LOGGED_IN
     */
    public static function mutateSetToCustomerDefaults(
        $unusedSrc,
        array $args
    ): string {
        $session = Mage::getSingleton("customer/session");
        $model = Mage::getSingleton("mageql_sales/quote");
        $modified = false;
        $customer = $session->isLoggedIn() ? $session->getCustomer() : null;
        $quote = $model->getQuote();
        $origShipping = $model->getShippingAddress();

        if( ! $customer) {
            return MageQL_Sales_Model_Quote_Address_Result::ERROR_NOT_LOGGED_IN;
        }

        if($origShipping && $customer->getDefaultBillingAddress() !== $customer->getDefaultShippingAddress()) {
            $origShipping->setSameAsBilling(false);
        }

        if($args["billing"]) {
            $defaultBillingAddress = $customer->getDefaultBillingAddress();

            if($defaultBillingAddress && $defaultBillingAddress->getId()) {
                $billingAddress = Mage::getModel("sales/quote_address")
                    ->importCustomerAddress($defaultBillingAddress);
                $quote->setBillingAddress($billingAddress);

                $modified = true;
            }
        }

        if($args["shipping"]) {
            $defaultShippingAddress = $customer->getDefaultShippingAddress();

            if($defaultShippingAddress && $defaultShippingAddress->getId()) {
                $shippingAddress = Mage::getModel("sales/quote_address")
                    ->importCustomerAddress($defaultShippingAddress);
                $shippingAddress->setSameAsBilling(false);
                $quote->setShippingAddress($shippingAddress);

                $modified = true;
            }
        }

        if($modified) {

            $model->saveSessionQuoteWithShippingRates();

            return MageQL_Sales_Model_Quote_Address_Result::SUCCESS;
        }

        return MageQL_Sales_Model_Quote_Address_Result::NOT_MODIFIED;
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function mutateSetBillingPlaceholder(
        $unusedSrc,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Sales_Model_Quote_Address_Result_Placeholder {
        $model = Mage::getSingleton("mageql_sales/quote");
        $quote = $model->getQuote();

        if( ! Mage::helper("mageql_sales")->getAllowPlaceholderBillingAddress($ctx->getStore())) {
            return new MageQL_Sales_Model_Quote_Address_Result_Placeholder(
                MageQL_Sales_Model_Quote_Address_Result_Placeholder::ERROR_DISABLED
            );
        }

        // Set dummy billing address if quote is virtual AND current billing address is empty
        if( ! $quote->isVirtual()) {
            return new MageQL_Sales_Model_Quote_Address_Result_Placeholder(
                MageQL_Sales_Model_Quote_Address_Result_Placeholder::ERROR_IS_PHYSICAL
            );
        }

        $old = $quote->getBillingAddress();

        if($old->getFirstname()) {
            return new MageQL_Sales_Model_Quote_Address_Result_Placeholder(
                MageQL_Sales_Model_Quote_Address_Result_Placeholder::NOT_MODIFIED,
                $old
            );
        }

        $defaultCountryId = $ctx->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY);
        $old->setAddressType(Mage_Sales_Model_Quote_Address::TYPE_BILLING)
            ->setFirstname("-")
            ->setLastname("-")
            ->setStreet(["-"])
            ->setPostcode("-")
            ->setTelephone("-")
            ->setCity("-")
            ->setCountryId($defaultCountryId);

        $quote->save();

        return new MageQL_Sales_Model_Quote_Address_Result_Placeholder(
            MageQL_Sales_Model_Quote_Address_Result_Placeholder::SUCCESS,
            $old
        );
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function mutateSetBilling($unusedSrc, array $args): MageQL_Sales_Model_Quote_Address_Result {
        $model = Mage::getSingleton("mageql_sales/quote");
        // We actually want to create it if it does not yet exist
        $billingAddress = $model->makeBillingAddress();
        $shippingAddress = $model->getShippingAddress();
        $addressData = $args["address"];

        // TODO: Load customer-address data here? Or do we make that a separate
        //       mutation, to reset to customer data? Or do we have a parameter
        //       which tells us to merge?

        if(self::setCommonFields($billingAddress, $addressData)) {
            // If we have modified it, also propagate to shipping if it is marked as same
            if($shippingAddress && $shippingAddress->getSameAsBilling()) {
                // Copy the info
                foreach(self::COPY_FIELDS as $field) {
                    $shippingAddress->setData($field, $billingAddress->getData($field));
                }

                // $shippingAddress->setSameAsBilling(0);
            }

            $model->saveSessionQuote();

            return new MageQL_Sales_Model_Quote_Address_Result(
                MageQL_Sales_Model_Quote_Address_Result::SUCCESS,
                $billingAddress
            );
        }

        return new MageQL_Sales_Model_Quote_Address_Result(
            MageQL_Sales_Model_Quote_Address_Result::NOT_MODIFIED,
            $billingAddress
        );
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function mutateSetShipping($unusedSrc, array $args): MageQL_Sales_Model_Quote_Address_Result  {
        $model = Mage::getSingleton("mageql_sales/quote");
        // We actually want to create it if it does not yet exist
        $shippingAddress = $model->makeShippingAddress();
        $addressData = $args["address"];

        // TODO: Load customer-address data here? Or do we make that a separate
        //       mutation, to reset to customer data? Or do we have a parameter
        //       which tells us to merge?

        if(self::setCommonFields($shippingAddress, $addressData)) {
            $shippingAddress->setSameAsBilling(false);

            $model->saveSessionQuoteWithShippingRates();

            return new MageQL_Sales_Model_Quote_Address_Result(
                MageQL_Sales_Model_Quote_Address_Result::SUCCESS,
                $shippingAddress
            );
        }

        return new MageQL_Sales_Model_Quote_Address_Result(
            MageQL_Sales_Model_Quote_Address_Result::NOT_MODIFIED,
            $shippingAddress
        );
    }

    public static function mutateUseBillingAsShipping(): MageQL_Sales_Model_Quote_Address_Result  {
        $model = Mage::getSingleton("mageql_sales/quote");
        $quote = $model->getQuote();
        $billingAddress = $model->makeBillingAddress();
        $shippingAddress = $model->getShippingAddress();
        $shippingMethod = $shippingAddress ? $shippingAddress->getShippingMethod() : null;

        // Create a new address object
        $newAddress = Mage::getModel("sales/quote_address");

        foreach($billingAddress->getData() as $key => $value) {
            if($value !== null && in_array($key, self::COPY_FIELDS)) {
                $newAddress->setData($key, $value);
            }
        }

        $newAddress->setSameAsBilling(true)
            ->setSaveInAddressBook(false)
            ->setCollectShippingRates(true);

        if($shippingMethod) {
            // TODO: Validate?
            $newAddress->setShippingMethod($shippingMethod);
        }

        if($shippingAddress) {
            $quote->removeAddress($shippingAddress);
        }

        $quote->setShippingAddress($newAddress);

        $model->saveSessionQuoteWithShippingRates();

        return new MageQL_Sales_Model_Quote_Address_Result(
            MageQL_Sales_Model_Quote_Address_Result::SUCCESS,
            $billingAddress
        );
    }

    public static function setCommonFields(Mage_Sales_Model_Quote_Address $address, array $data): bool {
        $modified = MageQL_Core_Model_Address::setCommonFields($address, $data);

        if($modified) {
            // Make sure we do not tie it to a customer address
            $address->setCustomerAddressId(null);
            $address->setSaveInAddressBook(false);
        }

        return $modified;
    }

    /**
     * Checks if the address is empty with respect to store defaults.
     */
    public static function isAddressEmpty(
        Mage_Customer_Model_Address_Abstract $address,
        Mage_Core_Model_Store $store
    ): bool {
        $defaultAddress = [
            "country_id" => $store->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY),
        ];
        $addrFields = array_flip(self::ADDRESS_DATA_FIELDS);
        $data = array_filter(array_diff(array_intersect_key($address->getData(), $addrFields), $defaultAddress));

        return count($data) === 0;
    }
}
