<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Sales_Model_Item_Bundle_Option {
    /**
     * @var array<string, float>
     */
    protected $children;

    /**
     * @var Mage_Catalog_Model_Product
     */
    protected $parent;

    /**
     * @var Mage_Bundle_Model_Option
     */
    protected $option;

    /**
     * @param Array<string, float> $children Map of product id => selection-qty
     */
    public function __construct(
        array $children,
        Mage_Catalog_Model_Product $parent,
        Mage_Bundle_Model_Option $option
    ) {
        $this->children = $children;
        $this->parent = $parent;
        $this->option = $option;
    }

    public function getType(): string {
        return $this->option->getType();
    }

    public function getTitle(): string {
        return $this->option->getTitle() ?: (string)$this->option->getDefaultTitle();
    }

    /**
     * @return array<array{ 0:Mage_Catalog_Model_Product, 1:float }>
     */
    public function getProducts(
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): array {
        /**
         * @var Mage_Bundle_Model_Resource_Selection_Collection
         */
        $collection = Mage::getResourceModel("bundle/selection_collection");

        $collection->setOptionIdsFilter([$this->option->getId()]);

        // No visibility filter here since bundle option selections can be invisible
        // URLs in case they are their own products
        $collection->addUrlRewrite();
        $collection->setPositionOrder();
        $collection->addStoreFilter($ctx->getStore());
        $collection->addMinimalPrice();
        $collection->joinPrices((int)$ctx->getStore()->getWebsiteId());
        $collection->addFieldToFilter("entity_id", ["in" => array_keys($this->children)]);

        $collection = MageQL_Catalog_Model_Product::applyAttributeSelects(
            $collection,
            $info,
            ["product"]
        );

        return array_map(function(Mage_Catalog_Model_Product $c): array {
            $c->addData([
                "parent_id" => $this->parent->getId(),
            ]);

            return [$c, $this->children[$c->getId()] ?? 1];
        }, $collection->getItems());
    }
}
