<?php

declare(strict_types=1);

namespace Wishlist;

use Spatie\Snapshots\MatchesSnapshots;

use Mage;
use MageQL_Sales_Model_BuyRequest;
use MageQL_Sales_Model_BuyRequest_Product_Configurable;
use MageQL_Sales_Model_BuyRequest_Product;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

require_once __DIR__."/Base.php";

class LoggedInTest extends Base {
    use MatchesSnapshots;

    public function setUp(): void {
        parent::setUp();

        $this->loginCustomer();
    }

    protected function loginCustomer(): void {
        MagentoManager::init();

        $store = Mage::app()->getStore();

        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());
    }

    protected function getWishlistData($data, $key) {
        $data = json_decode($data, true);

        $this->assertIsArray($data);
        $this->assertArrayHasKey("data", $data);
        $this->assertArrayHasKey("products", $data["data"]);
        $this->assertArrayHasKey("items", $data["data"][$key]);
        $this->assertIsArray($data["data"][$key]["items"]);

        $itemIds = [];

        $products = array_map(function($i) use(&$itemIds) {
            $this->assertIsArray($i);

            $itemIds[] = ($i["wishlist"] ?? [])["itemId"] ?? null;

            $i["wishlist"] = array_filter($i["wishlist"], function($k) { return $k !== "itemId"; }, ARRAY_FILTER_USE_KEY);

            return $i;
        }, $data["data"][$key]["items"]);

        return [
            "itemIds" => $itemIds,
            "products" => $products,
        ];
    }

    public function testProducts() {
        $wishlistItemIdSimple = $this->fetchWishlistItemId("test-simple");
        $wishlistItemIdConfig = $this->fetchWishlistItemId("test-config");

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            products {
                items {
                    sku
                    wishlist { selected, addedAt, itemId }
                }
            }
        }'));

        $data = $this->getWishlistData($res->getBody(), "products");

        $this->assertMatchesSnapshot($data["products"]);
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
        $this->assertEquals([
            null,
            null,
            null,
            $wishlistItemIdConfig,
            $wishlistItemIdSimple,
            null,
        ], $data["itemIds"]);
    }

    public function testProductDetail() {
        $wishlistItemId = $this->fetchWishlistItemId("test-simple");

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku(sku: "test-simple") {
              sku
              wishlist { selected, addedAt, itemId }
            }
        }'));

        $data = json_decode($res->getBody(), true);

        $this->assertIsArray($data);
        $this->assertArrayHasKey("data", $data);
        $this->assertArrayHasKey("productBySku", $data["data"]);
        $this->assertIsArray($data["data"]["productBySku"]);

        $item = $data["data"]["productBySku"];

        $this->assertEquals("test-simple", $item["sku"]);
        $this->assertArrayHasKey("wishlist", $item);
        $this->assertTrue($item["wishlist"]["selected"]);
        $this->assertEquals("2020-01-18T17:00:00Z", $item["wishlist"]["addedAt"]);
        $this->assertEquals($wishlistItemId, $item["wishlist"]["itemId"]);
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProductConfig() {
        $wishlistItemId = $this->fetchWishlistItemId("test-config");

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            productBySku(sku: "test-config") {
                sku
                wishlist { selected, addedAt, itemId }
            }
        }'));

        $data = json_decode($res->getBody(), true);

        $this->assertIsArray($data);
        $this->assertArrayHasKey("data", $data);
        $this->assertArrayHasKey("productBySku", $data["data"]);
        $this->assertIsArray($data["data"]["productBySku"]);

        $item = $data["data"]["productBySku"];

        $this->assertEquals("test-config", $item["sku"]);
        $this->assertArrayHasKey("wishlist", $item);
        $this->assertTrue($item["wishlist"]["selected"]);
        $this->assertEquals("2020-01-01T17:00:00Z", $item["wishlist"]["addedAt"]);
        $this->assertEquals($wishlistItemId, $item["wishlist"]["itemId"]);
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testAddSimple() {
        $this->clearWishlist();

        $store = Mage::app()->getStore();

        // Add simple product
        $product = Mage::getModel("catalog/product");
        $product->load($product->getIdBySku("test-simple"));

        $buyRequest1 = MageQL_Sales_Model_BuyRequest::fromProduct($product, $store);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            addToWishlist(buyRequest:'.json_encode($buyRequest1).') {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        // Assert the wishlist
        $wishlist = Mage::getModel("wishlist/wishlist");
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        $wishlist->loadByCustomer($customer);

        $this->assertEquals(1, $wishlist->getItemsCount());
        $this->assertEquals($product->getId(), $wishlist->getItemCollection()->getFirstItem()->getProductId());
    }

    public function testAddConfig() {
        $this->clearWishlist();

        $store = Mage::app()->getStore();
        $parent = Mage::getModel("catalog/product");
        $child = Mage::getModel("catalog/product");

        $parent->load($parent->getIdBySku("test-config"));
        $child->load($child->getIdBySku("test-config-child-2"));

        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $parent->getTypeInstance(true);

        $buyRequest2 = new MageQL_Sales_Model_BuyRequest_Product_Configurable($parent, $child, $instance->getConfigurableAttributes($parent));

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            addToWishlist(buyRequest:'.json_encode($buyRequest2).') {
                result
                wishlistItem { selected addedAt itemId }
            }
        }'));

        $data = json_decode($res->getBody(), true);

        $this->assertIsArray($data);
        $this->assertArrayHasKey("data", $data);
        $this->assertArrayHasKey("addToWishlist", $data["data"]);
        $this->assertIsArray($data["data"]["addToWishlist"]);

        $item = $data["data"]["addToWishlist"];

        $this->assertEquals("success", $item["result"]);
        $this->assertArrayHasKey("wishlistItem", $item);
        $this->assertTrue($item["wishlistItem"]["selected"]);
        $this->assertNotNull("2020-01-01T17:00:00Z", $item["wishlistItem"]["addedAt"]);

        $wishlistItemId = $this->fetchWishlistItemId("test-config");

        $this->assertEquals($wishlistItemId, $item["wishlistItem"]["itemId"]);
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        // Assert the wishlist
        $wishlist = Mage::getModel("wishlist/wishlist");
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        $wishlist->loadByCustomer($customer);

        $reqData = $buyRequest2->getBuyRequest();

        $reqData["product"] = $parent->getId();
        $reqData["qty"] = 1.0;
        $reqData["original_qty"] = 1;

        $this->assertEquals(1, $wishlist->getItemsCount());
        $this->assertEquals($parent->getId(), $wishlist->getItemCollection()->getFirstItem()->getProductId());
        $this->assertEquals($reqData, $wishlist->getItemCollection()->getFirstItem()->getBuyRequest()->getData());
    }

    public function testRemove() {
        // Try to remove item from wishlist
        $wishlistItemId = $this->fetchWishlistItemId("test-simple");
        $store = Mage::app()->getStore();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            removeFromWishlist(itemId:'.json_encode($wishlistItemId).') {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        // Assert the wishlist only contains the config
        $wishlist = Mage::getModel("wishlist/wishlist");
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        $parent = Mage::getModel("catalog/product");
        $child = Mage::getModel("catalog/product");

        $parent->load($parent->getIdBySku("test-config"));
        $child->load($child->getIdBySku("test-config-child-1"));

        $wishlist->loadByCustomer($customer);

        /**
         * @var Mage_Catalog_Model_Product_Type_Configurable
         */
        $instance = $parent->getTypeInstance(true);

        $buyRequest = new MageQL_Sales_Model_BuyRequest_Product_Configurable($parent, $child, $instance->getConfigurableAttributes($parent));
        $reqData = $buyRequest->getBuyRequest();

        $reqData["product"] = $parent->getId();
        $reqData["qty"] = 1.0;
        $reqData["original_qty"] = 1;

        $this->assertEquals(1, $wishlist->getItemsCount());
        $this->assertEquals($parent->getId(), $wishlist->getItemCollection()->getFirstItem()->getProductId());
        $this->assertEquals($reqData, $wishlist->getItemCollection()->getFirstItem()->getBuyRequest()->getData());
    }

    public function testFailAdd() {
        $store = Mage::app()->getStore();
        // Assert the wishlist
        $wishlist = Mage::getModel("wishlist/wishlist");
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        $wishlist->loadByCustomer($customer);

        $this->assertEquals(2, $wishlist->getItemsCount());

        $store = Mage::app()->getStore();
        $product = Mage::getModel("catalog/product");
        $product->load($product->getIdBySku("test-unassigned"));

        // Try to add wrong product
        $buyRequest = new MageQL_Sales_Model_BuyRequest_Product($product);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'mutation {
            addToWishlist(buyRequest:'.json_encode($buyRequest).') {
                result
                wishlistItem { selected addedAt itemId }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));

        // Assert the wishlist
        $wishlist = Mage::getModel("wishlist/wishlist");
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        $wishlist->loadByCustomer($customer);

        $this->assertEquals(2, $wishlist->getItemsCount());
    }

    public function clearWishlist() {
        $sqlQuery = "DELETE FROM wishlist_item WHERE product_id > 0";
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
        $stmt->execute();
    }
}
