<?php

declare(strict_types=1);

namespace Quote;

use Spatie\Snapshots\MatchesSnapshots;

use Mage;
use MageQL_Sales_Model_Product;
use MageQL_Sales_Model_Quote_Item;
use Varien_Object;
use MageQL_Sales_Model_BuyRequest;
use MageQL_Sales_Model_BuyRequest_Product_Configurable;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class ItemsTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testEmpty() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        product {
                            sku
                            attributes {
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                    }
                    shipping {
                        method {
                            description
                        }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testConfigurable() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");
        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));

        $request = new Varien_Object([
            "qty" => 1,
            "super_attribute" => [$attr->getId() => $attrBlack],
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal {
                            incVat
                        }
                        product {
                            sku
                            url
                            buyRequest
                            attributes {
                                name
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                        ... on QuoteItemConfigurable {
                            configOption {
                                attributes {
                                    attribute
                                    label
                                    value
                                }
                                product {
                                    sku
                                    buyRequest
                                    attributes {
                                        name
                                        shortDescription
                                        smallImage {
                                            src
                                        }
                                    }
                                }
                            }
                        }
                    }
                    isVirtual
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSimple() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal {
                            incVat
                        }
                        product {
                            sku
                            url
                            attributes {
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                    }
                    isVirtual
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testVirtual() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Virtual requires billing instead of shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal {
                            incVat
                        }
                        product {
                            sku
                            url
                            attributes {
                                shortDescription
                                smallImage {
                                    src
                                }
                            }
                        }
                    }
                    isVirtual
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddInvalid() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: "{", qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddSimple() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddSimpleChained() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }' ],
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddUnassigned() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-unassigned"));

        // Faked buy request since product is technically not saleable and won't
        // get a buyRequest assigned
        $buyRequest = json_encode([
            "p" => (int)$product->getId(),
        ]);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 1) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $this->assertEquals(false, $quote->hasItems());
    }

    public function testAddVirtual() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddVirtualChained() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }' ],
            [ "query" => 'query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddConfigurable() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");
        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));
        $attrs = $product->getTypeInstance(true)->getConfigurableAttributes($product);
        $child = $product->getTypeInstance(true)->getUsedProductCollection($product)
                ->addAttributeToSelect("*")
                ->addAttributeToFilter("color", $attrBlack)
                ->getFirstItem();

        $buyRequest = new MageQL_Sales_Model_BuyRequest_Product_Configurable($product, $child, $attrs);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddConfigVariant() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config-child-1"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 1) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        $this->assertEquals(false, $quote->hasItems());
    }

    public function testBuyRequest() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $simpleProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 2,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($simpleProduct, $request);

        // Now add a config
        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");
        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        // We have to use load, cannot use loadByAttribute or similar
        $configProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));

        $request = new Varien_Object([
            "qty" => 1,
            "super_attribute" => [$attr->getId() => $attrBlack],
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($configProduct, $request);
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    items {
                        itemBuyRequest
                        qty
                        product {
                            sku
                        }
                    }
                }
            }
        '));

        $data = json_decode($resp->getBody(), true);

        $this->assertEquals(2, count($data["data"]["quote"]["items"]));
        $first = json_decode($data["data"]["quote"]["items"][0]["itemBuyRequest"], true);
        $second = json_decode($data["data"]["quote"]["items"][1]["itemBuyRequest"], true);

        $this->assertArrayHasKey("i", $first);
        $this->assertEquals($first["p"], $simpleProduct->getId());
        $this->assertArrayHasKey("i", $second);
        $this->assertEquals($second["p"], $configProduct->getId());

        // $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleDefault() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-default"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        $childItem = null;

        foreach(Mage::getSingleton("checkout/session")->getQuote()->getAllItems() as $item) {
            if($item->getParentItemId()) {
                $childItem = $item;
            }
        }

        // Ensure the option title has been loaded for the child item option
        $this->assertStringContainsString("Include extra", $childItem->getOptionsByCode()["bundle_selection_attributes"]->getValue());

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                        ... on QuoteItemBundle {
                            bundleOptions {
                                type
                                title
                                products {
                                    qty
                                    product {
                                        sku
                                        name
                                        attributes {
                                            shortDescription
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleDefaultWithoutOptions() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-default"));

        $storeId = Mage::app()->getStore()->getId();
        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());
        $bundleOptions = array_values(array_filter(array_map(function($o) {
            return [
                "optionId" => $o->getId(),
            ];
        }, $product->getTypeInstance(true)->setStoreFilter($storeId, $product)->getOptions($product))));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            "query" => 'mutation ($buyRequest: ID!, $qty: Int!, $bundleOptions: [AddQuoteItemBundleOption!]!) {
                addQuoteItem(buyRequest: $buyRequest, qty: $qty, bundleOptions: $bundleOptions) {
                    result
                }
            }',
            "variables" => [
                "buyRequest" => $buyRequest,
                "qty" => 3,
                "bundleOptions" => $bundleOptions,
            ]
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleSelectWithMultipleOption() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-select"));

        $storeId = Mage::app()->getStore()->getId();
        $buyRequest = json_encode([ "p" => (int)$product->getId() ]);
        $bundleOptions = array_merge(...array_values(array_map(function($o) {
                return array_values(array_map(function($s) use($o) {
                    return [
                        "optionId" => $o->getId(),
                        "qty" => 1,
                        "selectionId" => $s->getSelectionId(),
                    ];
                }, Mage::getResourceModel("bundle/selection_collection")
                    ->setOptionIdsFilter([$o->getId()])
                    ->addStoreFilter(Mage::app()->getStore())
                    ->getItems()
                ));
        }, $product->getTypeInstance(true)->setStoreFilter($storeId, $product)->getOptions($product))));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            "query" => 'mutation ($buyRequest: ID!, $qty: Int!, $bundleOptions: [AddQuoteItemBundleOption!]!) {
                addQuoteItem(buyRequest: $buyRequest, qty: $qty, bundleOptions: $bundleOptions) {
                    result
                }
            }',
            "variables" => [
                "buyRequest" => $buyRequest,
                "qty" => 3,
                "bundleOptions" => $bundleOptions,
            ]
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleDefaultWithQty() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-default"));

        $storeId = Mage::app()->getStore()->getId();
        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());
        $bundleOptions = array_values(array_filter(array_map(function($o) {
            return [
                "optionId" => $o->getId(),
                "qty" => 2,
                "selectionId" => array_values(array_map(function($s) {
                        return $s->getSelectionId();
                    }, Mage::getResourceModel("bundle/selection_collection")
                        ->setOptionIdsFilter([$o->getId()])
                        ->addStoreFilter(Mage::app()->getStore())
                        ->getItems()
                    ))[0],
            ];
        }, $product->getTypeInstance(true)->setStoreFilter($storeId, $product)->getOptions($product))));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            "query" => 'mutation ($buyRequest: ID!, $qty: Int!, $bundleOptions: [AddQuoteItemBundleOption!]!) {
                addQuoteItem(buyRequest: $buyRequest, qty: $qty, bundleOptions: $bundleOptions) {
                    result
                }
            }',
            "variables" => [
                "buyRequest" => $buyRequest,
                "qty" => 3,
                "bundleOptions" => $bundleOptions,
            ]
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleSelectWithSelectionQty() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-select"));

        $storeId = Mage::app()->getStore()->getId();
        $buyRequest = json_encode([ "p" => (int)$product->getId() ]);
        $bundleOptions = array_values(array_filter(array_map(function($o) {
            return [
                "optionId" => $o->getId(),
                "qty" => 2,
                "selectionId" => array_values(array_map(function($s) {
                        return $s->getSelectionId();
                    }, Mage::getResourceModel("bundle/selection_collection")
                        ->setOptionIdsFilter([$o->getId()])
                        ->addStoreFilter(Mage::app()->getStore())
                        ->getItems()
                    ))[1],
            ];
        }, $product->getTypeInstance(true)->setStoreFilter($storeId, $product)->getOptions($product))));

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            "query" => 'mutation ($buyRequest: ID!, $qty: Int!, $bundleOptions: [AddQuoteItemBundleOption!]!) {
                addQuoteItem(buyRequest: $buyRequest, qty: $qty, bundleOptions: $bundleOptions) {
                    result
                }
            }',
            "variables" => [
                "buyRequest" => $buyRequest,
                "qty" => 3,
                "bundleOptions" => $bundleOptions,
            ]
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    addresses {
                        type
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        company
                        street
                        city
                        postcode
                        region {
                            code
                            name
                        }
                        country {
                            code
                        }
                        ... on QuoteAddressBilling {
                            isUsedAsShipping
                        }
                    }
                    grandTotal {
                        incVat
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                        ... on QuoteItemBundle {
                            bundleOptions {
                                type
                                title
                                products {
                                    qty
                                    product {
                                        sku
                                        name
                                        attributes {
                                            shortDescription
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testAddBundleSelectWithNoOptions() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-select"));

        $buyRequest = json_encode([ "p" => (int)$product->getId() ]);
        $bundleOptions = [];

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            "query" => 'mutation ($buyRequest: ID!, $qty: Int!, $bundleOptions: [AddQuoteItemBundleOption!]!) {
                addQuoteItem(buyRequest: $buyRequest, qty: $qty, bundleOptions: $bundleOptions) {
                    result
                }
            }',
            "variables" => [
                "buyRequest" => $buyRequest,
                "qty" => 3,
                "bundleOptions" => $bundleOptions,
            ]
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateQty() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        // We have to use load, cannot use loadByAttribute or similar
        $simpleProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 2,
        ]);
        $simpleItem = $quote->addProduct($simpleProduct, $request);

        // Now add a config
        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");
        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        // We have to use load, cannot use loadByAttribute or similar
        $configProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));

        $request = new Varien_Object([
            "qty" => 1,
            "super_attribute" => [$attr->getId() => $attrBlack],
        ]);

        $configItem = $quote->addProduct($configProduct, $request)->getParentItem();

        $storeId = Mage::app()->getStore()->getId();
        $bundleProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-default"));
        $bundleOptions = $bundleProduct->getTypeInstance(true)
            ->setStoreFilter($storeId, $bundleProduct)
            ->getOptions($bundleProduct);

        $request = new Varien_Object([
            "qty" => 1,
            "bundle_option" => array_combine(
                array_map(function($o) { return $o->getId(); }, $bundleOptions),
                array_map(function($o) {
                    return array_values(array_map(function($s) {
                        return $s->getSelectionId();
                    }, Mage::getResourceModel("bundle/selection_collection")
                        ->setOptionIdsFilter([$o->getId()])
                        ->addStoreFilter(Mage::app()->getStore())
                        ->getItems()
                    ));
                }, $bundleOptions)
            ),
            "bundle_option_qty" => array_combine(
                array_map(function($o) { return $o->getId(); }, $bundleOptions),
                array_map(function($o) { return 1; }, $bundleOptions)
            ),
        ]);

        $bundleItem = $quote->addProduct($bundleProduct, $request)->getParentItem();

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        $buyRequestSimple = MageQL_Sales_Model_BuyRequest::fromQuoteItem($simpleItem);
        $buyRequestConfig = MageQL_Sales_Model_BuyRequest::fromQuoteItem($configItem);
        $buyRequestBundle = MageQL_Sales_Model_BuyRequest::fromQuoteItem($bundleItem);

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                updateQuoteItemQty(itemBuyRequest: '.json_encode($buyRequestSimple).', qty: 4) {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                updateQuoteItemQty(itemBuyRequest: '.json_encode($buyRequestConfig).', qty: 2) {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                updateQuoteItemQty(itemBuyRequest: '.json_encode($buyRequestBundle).', qty: 3) {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateQtyHiddenBundle() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $storeId = Mage::app()->getStore()->getId();
        $bundleProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-bundle-hidden"));

        $bundleOptions = $bundleProduct->getTypeInstance(true)
            // Prepare store filter for correct option title data
            ->setStoreFilter($storeId, $bundleProduct)
            ->getOptions($bundleProduct);

        $request = new Varien_Object([
            "qty" => 1,
            "bundle_option" => array_combine(
                array_map(function($o) { return $o->getId(); }, $bundleOptions),
                array_map(function($o) {
                    return array_values(array_map(function($s) {
                        return $s->getSelectionId();
                    }, Mage::getResourceModel("bundle/selection_collection")
                        ->setOptionIdsFilter([$o->getId()])
                        ->addStoreFilter(Mage::app()->getStore())
                        ->getItems()
                    ));
                }, $bundleOptions)
            ),
            "bundle_option_qty" => array_combine(
                array_map(function($o) { return $o->getId(); }, $bundleOptions),
                array_map(function($o) { return 2; }, $bundleOptions)
            ),
        ]);

        $bundleItem = $quote->addProduct($bundleProduct, $request)->getParentItem();

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        $buyRequestBundle = MageQL_Sales_Model_BuyRequest::fromQuoteItem($bundleItem);

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                updateQuoteItemQty(itemBuyRequest: '.json_encode($buyRequestBundle).', qty: 3) {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
        ])));

        $quote = Mage::getModel("sales/quote")->load($quoteId);

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        // Ensure we did not duplicate quote items, should be one parent and one child:
        $this->assertEquals(2, count($quote->getAllItems()));
    }

    public function testRemove() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $simpleProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 2,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $simpleItem = $quote->addProduct($simpleProduct, $request);

        // Now add a config
        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");
        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        // We have to use load, cannot use loadByAttribute or similar
        $configProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));

        $request = new Varien_Object([
            "qty" => 1,
            "super_attribute" => [$attr->getId() => $attrBlack],
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $configItem = $quote->addProduct($configProduct, $request)->getParentItem();

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        $buyRequestSimple = MageQL_Sales_Model_BuyRequest::fromQuoteItem($simpleItem);
        $buyRequestConfig = MageQL_Sales_Model_BuyRequest::fromQuoteItem($configItem);

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                removeQuoteItem(itemBuyRequest: '.json_encode($buyRequestSimple).') {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                removeQuoteItem(itemBuyRequest: '.json_encode($buyRequestConfig).') {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateQtyAndId() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $simpleProduct = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 2,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $simpleItem = $quote->addProduct($simpleProduct, $request);

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        $unassigned = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-unassigned"));

        // Faked buy request since product is technically not saleable and won't
        // get a buyRequest assigned, and we construct an invalid buy-request
        // in the attempt to replace the product
        $buyRequestSimple = json_encode([
            "i" => $simpleItem->getId(),
            "p" => (int)$unassigned->getId(),
        ]);

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
            [ "query" => 'mutation {
                updateQuoteItemQty(itemBuyRequest: '.json_encode($buyRequestSimple).', qty: 4) {
                    result
                }
            }'],
            [ "query" => 'query {
                quote {
                    items {
                        qty
                        product {
                            sku
                        }
                    }
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
