<?php

declare(strict_types=1);

use Spatie\Snapshots\MatchesSnapshots;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

/**
 * @runTestsInSeparateProcesses
 * @preserveGlobalState disabled
 */
class ApicompatQuoteTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
        MagentoManager::initAdmin();

        // Got to reset these
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData("carriers/freeshipping/active", 1);
        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 100);
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();
        MagentoManager::initAdmin();

        // Got to reset these
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData("carriers/freeshipping/active", 1);
        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 100);

        throw $e;
    }

    public function testQuoteServiceSubmitAllException() {
        Config::addConfig([
            [ "name" => "global", "children" => [
                [ "name" => "events", "children" => [
                    [ "name" => "mageql_sales_place_order_result_type", "children" => [
                        [ "name" => "observers", "children" => [
                            [ "name" => "test_quote_service_submit_all", "children" => [
                                [ "name" => "type", "value" => "type" ],
                                [ "name" => "class", "value" => "mageql_sales/test_observer" ],
                                [ "name" => "method", "value" => "quoteSubmitAllAddErrors" ],
                            ]],
                        ]],
                    ]],
                    [ "name" => "sales_model_service_quote_submit_before", "children" => [
                        [ "name" => "observers", "children" => [
                            [ "name" => "test_quote_service_submit_all", "children" => [
                                [ "name" => "type", "value" => "singleton" ],
                                [ "name" => "class", "value" => "mageql_sales/test_observer" ],
                                [ "name" => "method", "value" => "quoteSubmitAllBeforeException" ],
                            ]],
                        ]],
                    ]],
                ]],
            ]],
        ]);

        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // Decrease so we get a zero subtotal checkout
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0);

        MagentoManager::reset();
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testQuoteServiceSubmitAllExceptionUnknown() {
        Config::addConfig([
            [ "name" => "global", "children" => [
                [ "name" => "events", "children" => [
                    [ "name" => "mageql_sales_place_order_result_type", "children" => [
                        [ "name" => "observers", "children" => [
                            [ "name" => "test_quote_service_submit_all", "children" => [
                                [ "name" => "type", "value" => "type" ],
                                [ "name" => "class", "value" => "mageql_sales/test_observer" ],
                                [ "name" => "method", "value" => "quoteSubmitAllAddErrors" ],
                            ]],
                        ]],
                    ]],
                    [ "name" => "sales_model_service_quote_submit_before", "children" => [
                        [ "name" => "observers", "children" => [
                            [ "name" => "test_quote_service_submit_all", "children" => [
                                [ "name" => "type", "value" => "singleton" ],
                                [ "name" => "class", "value" => "mageql_sales/test_observer" ],
                                [ "name" => "method", "value" => "quoteSubmitAllBeforeUnknownException" ],
                            ]],
                        ]],
                    ]],
                ]],
            ]],
        ]);

        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // Decrease so we get a zero subtotal checkout
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0);

        MagentoManager::reset();
        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        // Add the product and then modify the item to zero the price
        $quote->addProduct($product, $request)
            ->setOriginalCustomPrice(0)
            ->setCustomPrice(0)
            ->setIsSuperMode(true);

        // It is not a virtual quote, use shipping address for payment
        $quote->getShippingAddress()->setPaymentMethod("free");
        $quote->getPayment()->importData([
            "method" => "free",
        ]);
        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        MagentoManager::reset();
        MagentoManager::init();

        Mage::getSingleton("checkout/session")->setQuoteId($quoteId);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
            }
        }'));

        $this->assertStringContainsString("Internal server error", $resp->getBody());
        $this->assertStringContainsString("The unknown exception", $resp->getBody());
        $this->assertEquals(500, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderResultAddErrors() {
        Config::addConfig([
            [ "name" => "global", "children" => [
                [ "name" => "events", "children" => [
                    [ "name" => "mageql_sales_place_order_result_type", "children" => [
                        [ "name" => "observers", "children" => [
                            [ "name" => "test_quote_service_submit_all", "children" => [
                                [ "name" => "type", "value" => "type" ],
                                [ "name" => "class", "value" => "mageql_sales/test_observer" ],
                                [ "name" => "method", "value" => "quoteSubmitAllAddErrors" ],
                            ]],
                        ]],
                    ]],
                ]],
            ]],
        ]);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
query EnumerationValues {
  __type(name: "PlaceOrderResultType") {
    kind
    name
    description
    enumValues {
      name
      description
    }
  }
}
'));

        $this->assertStringContainsString("\"errorTheTestApiExceptionHappened\"", $resp->getBody());
        $this->assertStringContainsString("\"Error for test of api exception\"", $resp->getBody());
    }
}

class MageQL_Sales_Model_Test_Observer extends Mage_Core_Model_Abstract {
    public function quoteSubmitAllAddErrors($event) {
        $errors = $event->getErrors();

        $errors->setData("errorTheTestApiExceptionHappened", [
            "apiErrorCode" => 23,
            "description" => "Error for test of api exception",
        ]);
    }

    public function quoteSubmitAllBeforeException() {
        throw Crossroads_API_ResponseException::create(403, "Not permitted", null, 23);
    }

    public function quoteSubmitAllBeforeUnknownException() {
        // Unknown error code
        throw Crossroads_API_ResponseException::create(403, "The unknown exception", null, 123);
    }
}

