<?php

declare(strict_types=1);

use GraphQL\Deferred;
use GraphQL\Type\Definition\ResolveInfo;
use function MageQL\snakeCaseToCamel;

class MageQL_Sales_Model_Wishlist {

    const SUCCESS = "success";
    const SUCCESS_REMOVING_PRODUCT = "successRemovingProduct";
    const ERROR_ADDING_PRODUCT = "errorAddingProduct";
    const ERROR_REMOVING_PRODUCT = "errorRemovingProduct";
    const ERROR_NOT_LOGGED_IN = "errorNotLoggedIn";
    const ERROR_UNKNOWN_PRODUCT = "errorUnknownProduct";

    public static function registryKey(): string
    {
        return "mageql_sales_wishlist";
    }

    public static function instance(): self
    {
        $wishlist = Mage::registry(static::registryKey());

        if (!$wishlist) {
            $wishlist = new static();

            Mage::register(static::registryKey(), $wishlist);
        }

        return $wishlist;
    }

    /**
     * Clears this deferred container by removing the global instance.
     */
    public static function clear(): void
    {
        Mage::unregister(static::registryKey());
    }

    private function __construct()
    {
        // Empty
    }

    /**
     * @var Array<int, array{ selected:bool, added_at: ?string }>
     */
    protected $data = [];

    /**
     * @var Array<int>
     */
    protected $productIds = [];

    /**
     * Queues the given product to fetch wishlist data.
     */
    public function add(Mage_Catalog_Model_Product $product): self
    {
        if (!in_array((int) $product->getId(), $this->productIds)) {
            $this->productIds[] = (int) $product->getId();
        }

        return $this;
    }

    public function load(): self
    {
        if (empty($this->productIds)) {
            // Nothing new to load
            return $this;
        }

        $storeId = Mage::app()->getStore()->getId();
        $customerSession = Mage::getSingleton("customer/session");
        $customerId = 0;

        if ($customerSession->isLoggedIn()) {
            $customerId = $customerSession->getCustomerId();
        }

        if (!empty($customerId) && !empty($storeId)) {
            $db = Mage::getSingleton('core/resource')->getConnection('core_read');
            $select = $db->select()
                    ->from(["w" => "wishlist"], [])
                    ->join(['wi' => 'wishlist_item'], "wi.wishlist_id = w.wishlist_id", ["wi.product_id", "wi.added_at", "wi.wishlist_item_id"])
                    ->where("w.customer_id = ?", $customerId)
                    ->where("wi.store_id = ?", $storeId)
                    ->where("wi.product_id IN (?)", $this->productIds);
            $result = $db->fetchAssoc($select->__toString());

            if (!empty($result)) {
                $this->data += $result;
            }

            $this->productIds = [];
        }

        return $this;
    }

    /**
     * Fetches data
     *
     * @return ?array{ selected:bool, addedAt: ?string }
     */
    public function get(Mage_Catalog_Model_Product $product): ?array
    {
        return $this->data[(int) $product->getId()] ?? null;
    }

    public static function resolveIsProductInWishlist(Mage_Catalog_Model_Product $product): Deferred
    {
        $wishlist = MageQL_Sales_Model_Wishlist::instance();

        $wishlist->add($product);

        return new Deferred(function() use($product, $wishlist) {
            $wishlist->load();

            $data = $wishlist->get($product);
            $date = $data['added_at'] ?? null;
            $wishlistItemId = $data['wishlist_item_id'] ?? null;

            return [
                "selected" => $data ? true : false,
                "addedAt" => empty($date) ? null : gmdate("Y-m-d\TH:i:s\Z", strtotime($date)),
                "itemId" => $wishlistItemId
            ];
        });
    }

    /**
     * @param mixed $unusedSrc
     * @param array{page:int, pageSize:int, value:string}|array{page:int, pageSize:int, to:string, from:string} $args
     */
    public static function resolveProductsInWishlist(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): Mage_Catalog_Model_Resource_Product_Collection {
        $storeId = Mage::app()->getStore()->getId();
        $customerId = 0;

        /*
         *  If customer is not logged in, just run with customerId = 0, since that generates empty collection.
         *  This is to avoid errors down the line.
         */
        if (Mage::getSingleton('customer/session')->isLoggedIn()) {
            $customerId = Mage::getSingleton('customer/session')->getCustomer()->getId();
        }

        $collection = Mage::getModel('catalog/product')->getCollection();

        $subquery = new Zend_Db_Expr(sprintf("SELECT `w2`.`product_id` FROM `wishlist` `w1` JOIN `wishlist_item` `w2` ON `w2`.`wishlist_id` = `w1`.`wishlist_id` WHERE `w1`.`customer_id` = %d AND `w2`.`store_id` = %d", $customerId, $storeId));
        $collection->getSelect()->where("`e`.`entity_id` IN (?)", $subquery);

        /*
        if( ! empty($args["attributeFilter"])) {
            $collection = MageQL_Catalog_Model_Product::prepareProductAttributeFilters($collection, $args["attributeFilter"]);
        }

        if( ! empty($args["priceFilter"])) {
            $collection = MageQL_Catalog_Model_Product::prepareProductPriceFilters($collection, $args["priceFilter"]);
        }
        */

        // Just plain URLs here since we do not have a category
        $collection->addUrlRewrite();
        $collection->addAttributeToSort("name", Varien_Data_Collection::SORT_ORDER_ASC);

        return MageQL_Catalog_Model_Product::preparePaginatedProductCollection($collection, $args, $ctx);
     }

    /**
     * @param Mage_Wishlist_Model_Item|string $src
     * @return string
     */
    public static function resolveResult($src) {
        return is_string($src) ? $src : self::SUCCESS;
    }

    /**
     * @param Mage_Wishlist_Model_Item|string $src
     * @return ?array{ selected:bool, addedAt: string, itemId: string }
     */
    public static function resolveResultItem($src): ?array {
        if(is_string($src)) {
            return null;
        }

        return [
            "selected" => true,
            "addedAt" => gmdate("Y-m-d\TH:i:s\Z", strtotime($src->getAddedAt())),
            "itemId" => (string)$src->getId(),
        ];
    }

    /**
     * Add product to wishlist
     *
     * @param mixed $unusedSrc
     * @return Mage_Wishlist_Model_Item|string
     */
    public static function mutateAddToWishlist($unusedSrc, array $args, MageQL_Core_Model_Context $ctx) {
        try {
            if (! Mage::getSingleton('customer/session')->isLoggedIn()) {
                return self::ERROR_NOT_LOGGED_IN;
            }

            $customerId = Mage::getSingleton('customer/session')->getCustomer()->getId();
            $storeId = $ctx->getStore()->getId();

            // Extract frontend version of buyRequest
            $request = json_decode($args["buyRequest"], true, 3);
            $productId = (int)($request["p"] ?? 0);

            if (empty($productId)) {
                return self::ERROR_UNKNOWN_PRODUCT;
            }

            // Create Magento version of buyRequest
            $buyRequest = ["product" => $productId, "qty" => 1];
            $product = Mage::getModel('catalog/product');

            $product->setStoreId($storeId)->load($productId);

            if ( ! Mage::helper("mageql_catalog")->isProductVisible($product, $ctx->getStore())) {
                return self::ERROR_UNKNOWN_PRODUCT;
            }
            if ($product->getTypeId() === "configurable") {
                if(array_key_exists("a", $request)) {
                    $buyRequest["super_attribute"] = $request["a"];
                }
            }

            // Add product to wishlist
            $wishlist = Mage::getModel('wishlist/wishlist')->loadByCustomer($customerId, true);

            $result = $wishlist->addNewItem($product, new Varien_Object($buyRequest));

            if(is_string($result)) {
                throw new Exception($result);
            }

            $wishlist->save();

            return $result;
        }
        catch (Exception $ex) {
            Mage::logException($ex);

            return self::ERROR_ADDING_PRODUCT;
        }
    }

    /**
     * Remove product from wishlist
     *
     * @return String
     */
    public static function mutateRemoveFromWishlist($src, array $args, $ctx) {
        $wishlistItemId = $args["itemId"] ?? null;

        if(empty($wishlistItemId)) {
            return self::ERROR_UNKNOWN_PRODUCT;
        }

        if( ! Mage::getSingleton('customer/session')->isLoggedIn()) {
            return self::ERROR_NOT_LOGGED_IN;
        }

        $customerId = Mage::getSingleton('customer/session')->getCustomer()->getId();

        try {
            $sqlQuery = "
                SELECT wi.wishlist_item_id
                FROM wishlist_item wi
                JOIN wishlist w ON w.wishlist_id = wi.wishlist_id
                WHERE w.customer_id = :customerId AND wi.wishlist_item_id = :wishlistItemId
            ";
            $params = [
                "customerId" => $customerId,
                "wishlistItemId" => $wishlistItemId
            ];
            $result = Mage::getSingleton('core/resource')->getConnection('core_read')->fetchOne($sqlQuery, $params);

            if ( ! empty($result) && (int)$result === (int)$wishlistItemId) {
                $item = Mage::getModel('wishlist/item')->load($wishlistItemId);
                $item->delete();

                return self::SUCCESS;
            }
        }
        catch (Exception $ex) {
            Mage::logException($ex);

            return self::ERROR_REMOVING_PRODUCT;
        }

        return self::ERROR_UNKNOWN_PRODUCT;
    }
}
